/*____________________________________________________________________________
	Copyright (C) 2000 Networks Associates Technology, Inc.
	All rights reserved.

	$Id: pgpTokenLib.c,v 1.62.2.14 2001/08/11 02:24:29 ajivsov Exp $
____________________________________________________________________________*/

#include "pgpSDKBuildFlags.h"

#define PGP_TRACE_P11 1	 /* Extensive logging on the debug console */

#ifndef CRYPTOKIDLL
#define CRYPTOKIDLL "slbck.dll"
#endif

#include "windows.h"

#pragma pack(push, cryptoki, 1)
#include "pkcs11.h"
#pragma pack(pop, cryptoki)

#include "pgpConfig.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "pgpTokenLib.h"
#include "pgpMem.h"
#include "pgpErrors.h"
#include "pgpUsuals.h"

#define KEYIDLEN	    8
#define ID_PADDED_LEN   20
    /* For IE and old "Datakey RSA CSP" 20 is requirement.
       20 is a size Netscape/IE uses (SHA-1) for keys 
	   generated within the browser. 
	   However, 8 works with Netscape 4.7 and 6.0 and IE.
    */
#define ID_LEN_MAX   20	/* This is a maximum I saw */

#if ID_PADDED_LEN < KEYIDLEN || ID_PADDED_LEN > ID_LEN_MAX || KEYIDLEN > ID_LEN_MAX
#error 
#endif

#undef pgpAssert
#ifdef PGP_WIN32
#include <assert.h>
#define pgpAssert assert
#else
#define pgpAssert(x)
#endif

#define sCheckCache(tptr)

static CK_FUNCTION_LIST_PTR F;
static PGPToken token[MAXTOKENS];
static PGPBoolean is_CRYPTOKI_initialized = FALSE;
static int s_how_many_tokens = -1;	/* don't know */
static PGPBoolean enable_checking = TRUE;

static PGPUInt32 *now_slots = NULL;
static PGPUInt32 now_slots_allocated = 0;

static PGPByte CryptoKiDll[255] = CRYPTOKIDLL;

/*** FORWARD DECLARATIONS ***/

static void			   PKCS11ConstructTokenFields(PGPToken *tptr);
static PGPError		   PKCS11ConstructAllTokenFields(void);
static PGPError		   PKCS11PutPublicData(PGPToken *tptr, PGPByte *keyID,
						   PGPByte *pubKeyBlock, PGPSize size, 
						   pgpTokenDataInfo *di );
static PGPError		   PKCS11PutPrivate(PGPToken *tptr, PGPBoolean is_master, 
                            TCL_PRIVATE_KEY_DATA_LIST);

static PGPError		   PKCS11Delete(PGPToken *tptr, const PGPByte *keyID, PGPBoolean is_last );

static PGPByte *	   PKCS11GetPublicData(PGPToken *tptr, PGPByte *keyID,
							PGPSize *size, pgpTokenDataInfo **di);
static PGPTokenCertInfo * PKCS11GetCerts(PGPToken *tptr, PGPTokenKeyInfo *keyID,
							PGPInt32 *n);
static PGPByte *	   PKCS11GetCert(PGPToken *tptr, PGPTokenCertInfo *certInfo,
							PGPSize *size);

static PGPTokenKeyInfo*sGetPubKeyIDs(PGPToken *tptr, PGPSize *n, PGPBoolean is_dup);
static PGPTokenKeyInfo*PKCS11GetPubKeyIDs(PGPToken *tptr, PGPSize *n);

static PGPTokenKeyInfo*sGetPrivKeyIDs(PGPToken *tptr, PGPSize *n_out, PGPBoolean is_dup );
static PGPTokenKeyInfo*PKCS11GetPrivKeyIDs(PGPToken *tptr, PGPSize *n);

static PGPError        PKCS11Logout(PGPToken *tptr);
static PGPError		   PKCS11Login(PGPToken *tptr, char const *input,
							PGPSize length);
static PGPError		   PKCS11Sign(PGPToken *tptr, CK_BYTE *keyID, CK_BYTE *in,
							PGPSize isize, CK_BYTE *out,
							PGPSize osize);
static PGPError		   PKCS11Decrypt(PGPToken *tptr, CK_BYTE *keyID,
							CK_BYTE *in, PGPSize isize,
							CK_BYTE *out, PGPSize osize);
static PGPError		   PKCS11Destroy(PGPToken *tptr);
static PGPError		   PKCS11Init(PGPToken *tptr, PGPUInt32 whichslot,
							PGPBoolean haveslot, PGPInt32 tokenNum);
static PGPError		   PKCS11Keygen(PGPToken *tptr, PGPByte *ID,
							PGPSize modulusSize, PGPBoolean genMaster, PGPByte *modulus,
							PGPUInt32 *pubExp);
static PGPError		   PKCS11EditKeyList(PGPToken *tptr, PGPByte *origID,
							PGPByte *newID);
static PGPError		   PKCS11SetPIN(PGPToken *tptr, char *oldPIN,
							PGPSize oldPINsize, char *newPIN,
							PGPSize newPINsize);
static PGPError			PKCS11Wipe(PGPToken *tptr);
static PGPError         PKCS11GetInfo( PGPToken *tptr, 
                            PGPTokenInfo *tokenInfo );
static PGPError         PKCS11ImportX509(PGPToken *tptr, const PGPByte *keyID, 
                            const PGPByte *userID, PGPSize userID_len, 
                            const PGPByte *x509, PGPSize x509_len, 
                            const PGPByte *subjDER, PGPSize  subjDER_len, 
                            const PGPByte *issuerDER, PGPSize issuerDER_len,
                            const PGPByte *sn, PGPSize sn_len );

static PGPError         PKCS11PutKeyContainer( PGPToken *tptr,
                            const PGPByte *keyID, 
                            const PGPByte *uuid, PGPSize uuidSize );
static PGPError         PKCS11GetKeyContainer( PGPToken *tptr,
                            const PGPByte *keyID, 
                            PGPByte **uuidOut, PGPSize *uuidOutSize );

#if defined(PGP_DEBUG) && PGP_DEBUG != 0 && defined(PGP_TRACE_P11)
#include <stdarg.h>
void spgpTokenLibTrace(const char *file, int line, const char *fmt, ...)  {
	va_list marker;
	char s[1024];
	char s2[1024];
	const char *p = strrchr( file, '\\' );

	if( p )  
		file = p+1;

	va_start( marker, fmt );     
	vsprintf( s, fmt, marker );
	va_end( marker ); 
	sprintf( s2, "%s(%d): %s\n", file, line, s );
#ifdef PGP_WIN32	
	OutputDebugString( s2 );
#else
	printf( s2 );	/* TODO: logging for other platforms */
#endif
}
#define PGP_TOKEN_TRACE( s ) spgpTokenLibTrace( __FILE__, __LINE__, s )
#define PGP_TOKEN_TRACE1( s, p1 ) spgpTokenLibTrace( __FILE__, __LINE__, s, p1 )
#define PGP_TOKEN_TRACE2( s, p1,p2 ) spgpTokenLibTrace( __FILE__, __LINE__, s, p1,p2 )
#define PGP_TOKEN_TRACE3( s, p1,p2,p3 ) spgpTokenLibTrace( __FILE__, __LINE__, s, p1,p2,p3 )
#else
#define PGP_TOKEN_TRACE( s ) 
#define PGP_TOKEN_TRACE1( s, p1 ) 
#define PGP_TOKEN_TRACE2( s, p1,p2 ) 
#define PGP_TOKEN_TRACE3( s, p1,p2,p3 ) 
#endif

/* Check for the uninitialized values */
#define PGP_TOKEN_HANDLE_OK( handle ) \
	pgpAssert( (handle) != 0xcdcdcdcd && (handle) != 0xcccccccc && (handle) != (-1) );

#pragma pack(push, _pgpTokenKeyInfoPriv, 1)
typedef struct _pgpTokenKeyInfoPriv {
	PGPByte keyid[KEYIDLEN];

	PGPBoolean pgpData;	
	PGPUInt8 alg;		/* kPGPPublicKeyAlgorithm_RSA */

    PGPInt16 id_size;	/* CKA_ID, connecting related objects */

	CK_OBJECT_HANDLE handle;/* corresponding PKCS11 object 
							(PGP data object or PKCS11 public key) */

    PGPByte id[ID_LEN_MAX];
} pgpTokenKeyInfoPriv;
#pragma pack(pop, _pgpTokenKeyInfoPriv)

typedef struct _pgpTokenCertInfoPriv {
	PGPByte keyid[KEYIDLEN];    
	CK_OBJECT_HANDLE handle;    /* X509 cert object */
} pgpTokenCertInfoPriv;


/****** INTERNAL FUNCTIONS ********/

/* Labels are likely to be visible to the user in viewers/browsers.
   It is safe to change them only here */
const static CK_CHAR application[] =    "PGP";
static const CK_CHAR labelPrefix[] =    "PGP Key Data 0x";

static PGPByte certLabelPrefix[] =      "PGP Cert. ";
											/* + user identity + */
static PGPByte certLabelSuffix[] =      " 0x";
											/* + keyID */

static CK_CHAR labelPub[] =             "PGP Public Key";
static CK_CHAR labelPubSub[] =          "PGP Public Subkey";

static CK_CHAR labelPriv[] =            "PGP Private Key";
static CK_CHAR labelPrivSub[] =         "PGP Private Subkey";

typedef PGPByte pgpPubKeyDataLabel[sizeof(labelPrefix)+2*KEYIDLEN];

/* Given KeyID, outputs string representation of it into 'out'
   Returns the number of bytes written
*/
static unsigned 
sKeyIDToString( const PGPByte *keyID, PGPByte out[2*KEYIDLEN], PGPBoolean compat ) 
{
    int i;
    unsigned off=0;

    if( out == NULL )
        return KEYIDLEN * ( compat ? 1 : 2 );

    for( i = compat ? KEYIDLEN/2 : 0; i<KEYIDLEN; ++i )
	{
		PGPByte c = (keyID[i] >> 4) & 0xf;
		out[off++] = (c > 9) ? ('A'+c-10) : ('0'+c);
		c = keyID[i] & 0xf;
		out[off++] = (c > 9) ? ('A'+c-10) : ('0'+c);
	}

    return off;
}

/* Create label for public key data blob */
static void sKeyIDToPublicLabel( const PGPByte *keyID, CK_ULONG *size, pgpPubKeyDataLabel label )
{
	CK_ULONG off;

	memcpy( label, labelPrefix, sizeof(labelPrefix)-1 );
	off = sizeof(labelPrefix) - 1;
    off += sKeyIDToString( keyID, label + off, FALSE );

    *size = off;
}

/* Create label for PKCS11 cert. Caller must free it.
   size doesn't include '\0' character (but string has it) */

static PGPByte *
sKeyIDToP11CertLabel( const PGPByte *keyID, 
                const PGPByte *userID, PGPSize userID_len, 
                CK_ULONG *size )
{
    PGPSize max_size = sizeof(certLabelPrefix)-1;
    PGPByte *out, *p;

    max_size += userID_len;
    max_size += sizeof(certLabelSuffix)-1;
    max_size += sKeyIDToString( keyID, NULL, TRUE );

    p = out = malloc( max_size+sizeof('\0') );
    if( out == NULL )
        return NULL;

	memcpy( p, certLabelPrefix, sizeof(certLabelPrefix)-1 );
    p += sizeof(certLabelPrefix)-1; 
    
    memcpy( p, userID, userID_len );
    p += userID_len;

    memcpy( p, certLabelSuffix, sizeof(certLabelSuffix)-1 );
    p += sizeof(certLabelSuffix)-1;
    
    p += sKeyIDToString( keyID, p, TRUE );

    *p = '\0';
    *size = max_size;

    return out;
}


/* 
	This is used to find PKCS11 objects using PKCS11 template. 
	templSize is the size of the templ in bytes
	(therefore, use sizeof(templ)). 

    If in_buffer==NULL, then this function will malloc the handle buffer
	and return it. Caller must free returned value in this case. 

	If the in_buffer is not NULL, then the function will fill this buffer
	and return the pointer to this buffer. 
 */
static CK_OBJECT_HANDLE *
sFindP11Objs( PGPToken *tptr, 
 			  const CK_ATTRIBUTE *searchTempl, int templSize, int *n, 
			  CK_OBJECT_HANDLE *in_buffer, int in_buffer_n)
{
	CK_RV rv;
	const sizeSearch = templSize  / sizeof(*searchTempl);
	int howmany;
	int n_handles = 2; /* average number of expected handles */
	int n_handles_used=0;
	void *to_free=NULL;

	CK_OBJECT_HANDLE *handles = NULL; 

	*n = 0;

	if (!tptr->have_session)
	{
		pgpDebugMsg( "sFindP11Objs called with no smartcard session" );
		return NULL;
	}

	if( in_buffer == NULL )  {
		to_free = handles = (CK_OBJECT_HANDLE*)calloc( sizeof(CK_OBJECT_HANDLE), n_handles );
		if( handles == NULL )
			return NULL;
	}

	rv = F->C_FindObjectsInit(tptr->session, (CK_ATTRIBUTE *)searchTempl, sizeSearch);
	if (rv != CKR_OK)
	{
		pgpDebugMsg( "C_FindObjectsInit failed" );
		free( to_free );
		return NULL;
	}

	if( in_buffer == NULL )  {	/* buffer is not specified */

		pgpAssert( handles != NULL );

		/* There is no way to know how many handles will be found in advance. 
		   So we use the trial method. */
		while(TRUE) {
			howmany = 0;
			rv = F->C_FindObjects(tptr->session, handles+n_handles_used, n_handles-n_handles_used, &howmany);
			if (rv != CKR_OK)  {
				PGP_TOKEN_TRACE( "C_FindObjects failed" );
				free( handles );
				return NULL;
			}
			n_handles_used += howmany;

			/* make sure we read all of them */
			if( n_handles > n_handles_used )  
				break;

			n_handles *= 2;
			to_free = handles = realloc( handles, sizeof(CK_OBJECT_HANDLE)*n_handles );
			if( to_free == NULL )  {
				n_handles_used = 0;
				break;
			}
		} 
		howmany = n_handles_used;
	}
	else  {		/* buffer is given */
		handles = in_buffer;

		rv = F->C_FindObjects(tptr->session, in_buffer, in_buffer_n, &howmany);
		if (rv != CKR_OK)
		{
			pgpDebugMsg( "C_FindObjects failed" );
			howmany = 0;
		}
	}

	/* Always finalize */
	rv = F->C_FindObjectsFinal(tptr->session);
	if (rv != CKR_OK)
	{
		pgpDebugMsg( "C_FindObjectsFinal failed" );
		free( to_free );
		return NULL;
	}

	if( howmany == 0 )  {
		free( to_free );
		PGP_TOKEN_TRACE( "sFindP11Objs call returns NULL" );
		return NULL;
	}

	PGP_TOKEN_TRACE1( "sFindP11Objs call returns with data: %d items", howmany );

	*n = howmany;
	return handles;
}

/* 
	Given PKCS11 object with CKA_ID attribute, 
	returns the handle to the related object, matching the 'templ' 
	of the 'size'.
	We assume that input object and the object to be found are 
	'connected' by CKA_ID.

	First row of the 'templ' must be the CKA_ID attribute. 
	Input object must have CKA_ID attribute.
	First matched object will be returned.
	output will have output->id and output->id_size initialized. 
 */
PGPBoolean static 
sObjToRelatedObj( PGPToken *tptr, 
			const pgpTokenKeyInfoPriv *input, 
			CK_ATTRIBUTE *templ, int templ_size, 
			pgpTokenKeyInfoPriv *output )  
{
	CK_RV rv;
	int howmany;

	const PGPBoolean isID = ( input->id_size > 0 );
	PGPByte *pID = (PGPByte *)( isID ? input->id : NULL );
	int size = input->id_size;

	memcpy( output, input, sizeof(*output) );

	pgpAssert( !isID || input->id[0] );
	pgpAssert( templ[0].type == CKA_ID );

	if( !isID ) {	/* ID is not specified, PKCS11 handle must */
		CK_ATTRIBUTE pubTemplate[] =  {
			{ CKA_ID, NULL, 0 }
		};

		pgpAssert( size == 0 );
		pgpAssert( input->handle );
	
		rv = F->C_GetAttributeValue(tptr->session, input->handle, pubTemplate, 1);
		if (rv != CKR_OK) {
			pgpDebugMsg( "C_GetAttributeValue failed" );
			return FALSE;
		}
		
		pID = malloc( pubTemplate[0].ulValueLen );
		if( pID == NULL )
			return FALSE;

		size = pubTemplate[0].ulValueLen;
		pubTemplate[0].pValue = pID;
		
		rv = F->C_GetAttributeValue(tptr->session, input->handle, pubTemplate, 1);
		if (rv != CKR_OK) {
			free(pID);
			pgpDebugMsg( "C_GetAttributeValue failed" );
			return FALSE;
		}

		PGP_TOKEN_TRACE("sObjToRelatedObj: ID was not given");
	}
#if PGP_DEBUG
	else
		PGP_TOKEN_TRACE("sObjToRelatedObj: ID was given");
#endif


	/* Now we certainly know ID of the desired object */
	{
		CK_OBJECT_HANDLE handle;

		templ[0].pValue = (void*)pID;
		templ[0].ulValueLen = size;
		
		if( sFindP11Objs( tptr, templ, templ_size, &howmany, &handle, 1 ) == NULL )
		{
			pgpDebugMsg( "Object not found although in keyid list" );
			if( !isID )
				free( pID );
			return FALSE;
		}
		pgpAssert( howmany == 1 );

		output->handle = handle;

		if( size > sizeof(output->id) )
			size = sizeof(output->id);

		/* Store ID */
		output->id_size = size;
		memcpy( output->id, templ[0].pValue, size );
	}

	if( !isID )
		free( pID );

	return TRUE;
}

/* Get private key info, corresponding to the public key */
PGPBoolean static 
sPubToPriv( PGPToken *tptr, 
			const pgpTokenKeyInfoPriv *pub, 
			pgpTokenKeyInfoPriv *priv )  
{
	CK_OBJECT_CLASS privClass = CKO_PRIVATE_KEY;
	CK_BBOOL ktTrue = TRUE;
	CK_KEY_TYPE keyType = CKK_RSA;
	CK_ATTRIBUTE privTemplate[] =
	{
		{ CKA_ID, NULL, 0 },	/* must be the first */
		{ CKA_CLASS, &privClass, sizeof(privClass) },
		{ CKA_KEY_TYPE, &keyType, sizeof(keyType) },
		{ CKA_TOKEN, &ktTrue, 1 }
	};

	return sObjToRelatedObj( tptr, pub, 
		privTemplate, sizeof(privTemplate), priv );
}

#if PGP_DEBUG
int sCacheIsSorted(const pgpTokenKeyInfoPriv *cache, int n)  {
	int i; 
	int j;
	unsigned smallest;

	for( i=0; i<n; i++ )  {
		smallest = *(unsigned*)cache[i].keyid;

		for( j=i+1; j<n; j++ )  {
			if( smallest > *(unsigned*)cache[j].keyid )  {
				PGP_TOKEN_TRACE2("Sorting problem: [%d] > [%d]", i, j );
				return 0;
			}
		}
	}

	return 1;
}
#endif

static PGPBoolean sKeyIDToKeyInfoFromCache(
				PGPToken *tptr, const PGPByte *keyID,
				const pgpTokenKeyInfoPriv *cache, int n, 
				pgpTokenKeyInfoPriv *ki, PGPBoolean is_private )
{
	int i;
	const unsigned keyID_u = *(unsigned*)keyID;

	int start_i=0;
	int end_i;

	PGPBoolean found=FALSE;

	if( ki )
		memset( ki, 0, sizeof(*ki) );

	if( !n )
		return FALSE;

	pgpAssert( sCacheIsSorted(cache, n) );

	end_i = n-1;

	pgpAssert( keyID_u != 0 );

	/* Perform binary search */
	while( end_i - start_i > 1 )  {
		const middle = start_i + (end_i - start_i) / 2;
		const unsigned u = *(unsigned*)cache[middle].keyid;

		if( keyID_u > u )  {
			start_i = middle;
		}
		else if( keyID_u < u )  {
			end_i  = middle;
		}
		else  {
			break;
		}

		pgpAssert( end_i >= start_i );
	}

	pgpAssert( start_i <= end_i );

	for (i = start_i ; i <= end_i; i ++)  {
		if( memcmp(keyID, cache[i].keyid, sizeof(cache[i].keyid))==0 ) {
			found = TRUE;
			break;
		}
	}

	if( !found )  {
		PGP_TOKEN_TRACE1( "sKeyIDToKeyInfoFromCache(keyID=%x): not found", *(unsigned*)keyID );
		return FALSE;
	}

	if( ki )  {
		if( is_private )  {
			PGPBoolean ret;
			
			ret = sPubToPriv( tptr, cache+i, ki );
			pgpAssert( ret );
		}
		else   {
			memcpy( ki, cache+i, sizeof(*ki) );
			PGP_TOKEN_HANDLE_OK( ki->handle );
		}
	}

	pgpAssert( ! ki || memcmp( ki->keyid, keyID, sizeof(ki->keyid) )==0 );

	return TRUE;
}


/*
  Return TRUE if the given token holds the given key.
  If ki is not NULL, fill it with information about that key.
  Two cases possible in the latter case:
  we need handle to the CKO_PUBLIC_KEY (is_private=FALSE) or
  we need handle to the CKO_SECRET_KEY key (is_private=TRUE)
 */
static PGPBoolean sKeyIDToPrivKeyInfo(PGPToken *tptr, const PGPByte *keyID,
				pgpTokenKeyInfoPriv *ki, PGPBoolean is_private)
{
	PGPUInt32 n;
	pgpTokenKeyInfoPriv *block;
	const unsigned keyID_u = *(unsigned*)keyID;

	PGP_TOKEN_TRACE2( "sKeyIDToPrivKeyInfo(keyID=%x, is_private=%d)", 
		*(unsigned*)keyID, is_private );

	/* Always look for details in priv key info, because it is the largest set */
	block = (pgpTokenKeyInfoPriv *)sGetPrivKeyIDs(tptr, &n, FALSE);

#if PGP_DEBUG
	if( !block || !n )
		PGP_TOKEN_TRACE("No keys in cache");
#endif

	return sKeyIDToKeyInfoFromCache( tptr, keyID, block, n, ki, is_private );
}


/*
	Returns handle to the "PGP Public Key Data", which is a set of signed UserIDs, 
	without the PGP Public Key packet. 
*/
static PGPBoolean sKeyIDToPubKeyInfo(PGPToken *tptr, const PGPByte *keyID,
								 pgpTokenKeyInfoPriv *ki )
{
	PGPUInt32 n = 0;
	pgpTokenKeyInfoPriv *block;

	PGP_TOKEN_TRACE1( "sKeyIDToPubKeyInfo(keyID=%x)", *(unsigned*)keyID );

	block = (pgpTokenKeyInfoPriv *)sGetPubKeyIDs(tptr, &n, FALSE);

	return sKeyIDToKeyInfoFromCache( tptr, keyID, block, n, ki, FALSE );
}

/* Returns content of CKA_VALUE for the given handle */
static PGPByte *
sGetP11Value( PGPToken *tptr, CK_OBJECT_HANDLE handle, PGPSize *size )  {
	PGPByte *value;
	CK_RV rv;
	CK_ATTRIBUTE dataTemplate = { CKA_VALUE, NULL, 0 };
    int mapSize;
	
    PGPByte *pubKeyPktOut=NULL;
	int len_out=0;

	PGP_TOKEN_TRACE("sGetP11Value");

	rv = F->C_GetAttributeValue(tptr->session, handle, &dataTemplate, 1 );
	if (rv != CKR_OK)
	{
	    pgpDebugMsg( "C_GetAttributeValue failed" );
		return NULL;
	}
		
	mapSize = dataTemplate.ulValueLen;
	if( mapSize == 0 )
		return NULL;

	value = malloc( mapSize );
	pgpAssert(value);

    dataTemplate.pValue = value;

	rv = F->C_GetAttributeValue(tptr->session, handle, &dataTemplate, 1);
	if (rv != CKR_OK)
	{
	    pgpDebugMsg( "C_GetAttributeValue failed" );
		free(value);
		value = NULL;
	}

    *size = mapSize;
    return value;
}

/* 
	Create PGP Public Key Packet from the PKCS11 CKO_PUBLIC_KEY object.
	Caller must free di.
*/
static PGPError 
sGetP11PubKey( PGPToken *tptr, PGPByte *keyID, pgpTokenPubKeyStub *keyStub,
			   pgpTokenDataInfo **di )  
{
	CK_RV rv;
	pgpTokenKeyInfoPriv ki; 

	*di = NULL;

	/* Try to find CKO_PUBLIC_KEY */
	sKeyIDToPrivKeyInfo( tptr, keyID, &ki, FALSE );

	if( ki.handle == (CK_OBJECT_HANDLE)NULL )  {
		PGP_TOKEN_TRACE("No PKCS11 public key objects found");
		return kPGPError_NoErr;
	}

	/* Here we have the handle, referencing the CKO_PUBLIC_KEY object */
	{
        PGPByte *buf;
		PGPByte *mod_buf;
		PGPByte *exp_buf;
		PGPUInt32 mod_size;
		PGPUInt32 exp_size;

		CK_ATTRIBUTE dataTmpl[] = 
		{
			{ CKA_MODULUS,			NULL, 0 },
			{ CKA_PUBLIC_EXPONENT,	NULL, 0 }
		};

		rv = F->C_GetAttributeValue(tptr->session, ki.handle, dataTmpl, 2);
		
		mod_size = dataTmpl[0].ulValueLen;
		exp_size = dataTmpl[1].ulValueLen;

		buf = calloc( sizeof(pgpTokenDataInfo) + mod_size + exp_size, 1 );
		if( IsNull( buf ) )
			return kPGPError_OutOfMemory;

        exp_buf = buf + sizeof(pgpTokenDataInfo);   /* Exponent first */
		mod_buf = exp_buf + exp_size;               /* Modulus next */

		dataTmpl[0].pValue = mod_buf;
		dataTmpl[1].pValue = exp_buf;

		rv = F->C_GetAttributeValue(tptr->session, ki.handle, dataTmpl, 2);
		if (rv != CKR_OK) {
			pgpDebugMsg( "C_GetAttributeValue failed" );
			free( buf );
			return kPGPError_SmartCardError;
		}

        {
            pgpTokenDataInfo *p = (pgpTokenDataInfo*)buf;
            p->alg = kPGPPublicKeyAlgorithm_RSA;
            p->exp_size = exp_size;
            p->mod_size = mod_size;

            memcpy( &(p->pubKeyStub), keyStub, sizeof(p->pubKeyStub) );
        }

        *di = (pgpTokenDataInfo *)buf;
	}

	return kPGPError_NoErr;
}


/* Get missing key info from the given partially defined key object
   
   buffer is temporary buffer, allocated/reallocated in this routine 
*/
static PGPBoolean 
sGetAllKeyInfo( PGPToken *tptr, 
				const pgpTokenKeyInfoPriv *input, 
				pgpTokenKeyInfoPriv *output,
				void **buffer, int *buffer_size )
{
	CK_RV rv;

	const CK_OBJECT_HANDLE handle = input->handle;
	const PGPBoolean isID = (PGPBoolean)input->id_size;
	const PGPBoolean isKeyID = *(unsigned*)input->keyid;

	CK_ATTRIBUTE modTemplate[] =
	{
		{ CKA_MODULUS, NULL, 0 },
		{ CKA_LABEL, NULL, 0 },
		{ CKA_ID, NULL, 0 }		/* Must be last */
	};
	int modTemplateSize = sizeof(modTemplate)/sizeof(modTemplate[0]);

	PGPByte *b = *buffer;
	int b_size = *buffer_size;
	int cur_len_mod, cur_len_id, cur_len_label, cur_len_sum;
	PGPByte *label;
	const PGPByte *id;

	pgpTokenKeyInfoPriv *p_out_temp=NULL;
	int p_out_temp_size=0;

	if( handle == (CK_OBJECT_HANDLE)NULL )  {
		memset( output, 0, sizeof(*output) );
		pgpAssert( 0 );
		return FALSE;
	}

	PGP_TOKEN_HANDLE_OK( handle );

	memcpy( output, input, sizeof(*output) );

	if( isID )  {
		if( isKeyID )  {
			PGP_TOKEN_TRACE("sGetAllKeyInfo: object already complete");
			return TRUE;	/* We know everything */
		}
		modTemplateSize --;		/* We know ID */
	}

	rv = F->C_GetAttributeValue(tptr->session, handle, modTemplate, modTemplateSize);
	if (rv != CKR_OK) {
		pgpDebugMsg( "C_GetAttributeValue failed: no MODULUS" );
		return FALSE;
	}

	cur_len_mod = modTemplate[0].ulValueLen;
	cur_len_label = modTemplate[1].ulValueLen;

	if( isID )
		cur_len_id = 0;
	else
		cur_len_id = modTemplate[2].ulValueLen;

	cur_len_sum = cur_len_mod + cur_len_id + cur_len_label;

	pgpAssert( cur_len_mod > 8 );
	if( cur_len_sum > b_size )  {
		b = (b==NULL) ? 
			malloc( cur_len_sum ) :
		    realloc( b, b_size + cur_len_sum );

		if( b == NULL )  {
			*buffer = NULL;
			*buffer_size = 0;
			return FALSE; /* kPGPError_OutOfMemory */;
		}

		b_size += cur_len_sum;

		*buffer = b;
		*buffer_size = b_size;
	}

	modTemplate[0].pValue = b;
	label = modTemplate[1].pValue = b + cur_len_mod;
	if( isID )  
		id = input->id;
	else
		id = modTemplate[2].pValue = label + cur_len_label;

	rv = F->C_GetAttributeValue(tptr->session, handle, modTemplate, modTemplateSize);
	if (rv != CKR_OK)  {
		pgpDebugMsg( "C_GetAttributeValue failed" );
		return FALSE;
	}

    /* Store CKA_ID for later */
	if( !isID )  {  /* ID was not given */
        const label_longer = (cur_len_id >= sizeof(output->id));

        memcpy( output->id, id, 
            label_longer ? sizeof(output->id) : cur_len_id );

        if( !label_longer )  {	
            memset( output->id+cur_len_id, 0, sizeof(output->id)-cur_len_id );
            output->id_size = cur_len_id;
        }
#if PGP_DEBUG
        else if( cur_len_id == sizeof(output->id) ) {
            output->id_size = sizeof(output->id);
        }
#endif
        else  {
            pgpAssert(0);
            /* We should work in this case, but I have never seen this. */
            output->id_size = sizeof(output->id);
        }
    }

	output->pgpData = TRUE;

    /* Compare labels to see if this is a PGP token */
	if( (cur_len_id > ID_LEN_MAX) /* never seen it */ ||
        ( ( cur_len_label != sizeof(labelPub)-1 || 
		    memcmp( label, labelPub, sizeof(labelPub)-1 )!=0 ) &&
          ( cur_len_label != sizeof(labelPubSub)-1 || 
		    memcmp( label, labelPubSub, sizeof(labelPubSub)-1 )!=0 ) ) )
	{
		/* not a PGP token -- take last KEYIDLEN bytes as keyid */
		id = (PGPByte*)b+cur_len_mod - KEYIDLEN; 
		PGP_TOKEN_TRACE1( "Not a PGP object keyid=%x", *(unsigned*)id );
		output->pgpData = FALSE;
	}

	memcpy( output->keyid, id, KEYIDLEN);
	output->alg = kPGPPublicKeyAlgorithm_RSA;	/* for now */

	PGP_TOKEN_HANDLE_OK( output->handle );
	
	return TRUE;
}

/*
	Get an array of KeyIDs for which we have PGP Public Key Data object. 

	Go through all PGP Public Key CKO_DATA objects and collect 
	KeyIDs. 
*/
static PGPTokenKeyInfo* sGetPubKeyIDsFromPgpData(
				PGPToken *tptr, 
				/* PGPTokenKeyInfo *p, int p_n,  */
				PGPSize *n)
{
	CK_RV rv;
	CK_OBJECT_HANDLE *handles = NULL;
	int howmany;

	pgpPubKeyDataLabel value;
	const int prefixLen = sizeof(labelPrefix) -1;
	CK_CHAR *keyIdHead = value + prefixLen;

	const CK_OBJECT_CLASS obj = CKO_DATA;
	CK_ATTRIBUTE searchTemplate[] =
	{
		{ CKA_CLASS, (CK_OBJECT_CLASS *)&obj, sizeof(obj) },
		{ CKA_APPLICATION, (CK_CHAR *)application, sizeof(application) - 1 }
	};

	CK_ATTRIBUTE dataTemplate[] = 
	{
		{ CKA_LABEL, &value, sizeof(value)-1 }
	};

	PGPUInt32 sizeData = sizeof(dataTemplate) / sizeof(dataTemplate[0]);

	pgpTokenKeyInfoPriv *out=NULL;
	int out_size = 0;

	int i;

	PGP_TOKEN_TRACE("sGetPubKeyIDsFromPgpData");

	*n = 0;
	handles = sFindP11Objs( tptr, searchTemplate, sizeof(searchTemplate), &howmany, NULL, 0 );
	if( handles == NULL )
		return NULL;

	for( i=0; i<howmany; i++ )  {
		memset( value, 0, sizeof(value) );

		rv = F->C_GetAttributeValue(tptr->session, handles[i], dataTemplate, sizeData);
		if (rv != CKR_OK)  {
			PGP_TOKEN_TRACE( "C_GetAttributeValue falied" );
			break;
		}

		if( strncmp( value, labelPrefix, prefixLen ) == 0 )  {
			PGPByte *p;
			PGPByte *in = keyIdHead;

			out = (out==NULL) ? 
				malloc( sizeof(pgpTokenKeyInfoPriv) ) : realloc( out, (out_size+1)*sizeof(pgpTokenKeyInfoPriv));

			/* Convert hex string to bytes */
			for( p = out[out_size].keyid; 
                    p<out[out_size].keyid + KEYIDLEN; p++ )  {
				PGPByte b = *in;
				*p = ((b  >= 'A') ?  (b -'A'+10) : (b - '0')) << 4;
				b = *(++in);
				*p |= (b  >= 'A') ?  (b -'A'+10) : (b - '0');
				in ++;
			}

			/* Store handles for optimization */
			out[out_size].handle = handles[i];

            /* In this case PKCS11 CKA_ID == KeyID. 
               (certs of this key, as well as a private key 
			   have the same CKA_ID) */
            out[out_size].id_size = KEYIDLEN;
            memcpy( out[out_size].id, out[out_size].keyid, KEYIDLEN );

			out[out_size].pgpData = TRUE;
			out[out_size].alg = kPGPPublicKeyAlgorithm_RSA;	/* for now */

			out_size ++;
		}
	}

	free(handles);

	if( rv != CKR_OK || out == NULL )  {
		free(out);
		out = NULL;
	}

	if( out == NULL )  
		return NULL;

	*n = out_size;
	return (PGPTokenKeyInfo*)out;
}

/* Collect keyIDs from the PKCS11 cert objects 

   This is intended for the browser-generated keys ONLY. 
   We assume here that there is ONLY one X509 certificate
   per public key.

   p is input buffer, allocated with malloc of p_n pgpTokenKeyInfoPriv 
   elements of. This function will reallocs p and returns it.
   If return is NULL, p is freed. 

   Requires public key id cash to be present. 
*/
static PGPTokenKeyInfo* 
sGetPubKeyIDsFromCerts(	PGPToken *tptr, 
						PGPTokenKeyInfo *p, int p_n, 
						int *n_out)
{
	CK_ATTRIBUTE dataTemplate[] = 
	{
		{ CKA_VALUE, NULL, 0 }
	};

	CK_OBJECT_CLASS certClass = CKO_CERTIFICATE;
	CK_CERTIFICATE_TYPE certType = CKC_X_509;
		CK_ATTRIBUTE certTemplate[] =
	{
		{ CKA_CLASS, &certClass, sizeof(certClass) },
		{ CKA_CERTIFICATE_TYPE, &certType, sizeof(certType) }
	};

	CK_OBJECT_CLASS pubClass = CKO_PUBLIC_KEY;
	CK_ATTRIBUTE pubTemplate[] =
	{
		{ CKA_ID, NULL, 0 },	/*must be the first*/
		{ CKA_CLASS, &pubClass, sizeof(pubClass) },
	};


	CK_OBJECT_HANDLE *handles = NULL;
	int howmany;
	int i;

	int n = (p==NULL ? 0 : p_n);

	void *data_buf=NULL;
	int data_buf_size=0;

	PGP_TOKEN_TRACE("sGetPubKeyIDsFromCerts");

	*n_out = 0;

	/* Find all cert. objects */
	handles = sFindP11Objs( tptr, certTemplate, sizeof(certTemplate), &howmany, NULL, 0 );
	if( handles == NULL )  
		return p;

	for( i=0; i<howmany; i++ )  {
		/* Find related public key object */
		pgpTokenKeyInfoPriv cert;
		pgpTokenKeyInfoPriv pub; 

		memset( &cert, 0, sizeof(cert) );
		cert.handle = handles[i];

		if( !sObjToRelatedObj( tptr, &cert, pubTemplate, sizeof(pubTemplate), &pub ) )  
		{
			continue;	/* ignore this object */
		}

		/* Now, see if this certificate is on the PGP key. In that case
		   pub.id will be a keyid as well */
		if( sKeyIDToKeyInfoFromCache( tptr, pub.id, (pgpTokenKeyInfoPriv*)p, p_n, NULL, FALSE ) )  {
			PGP_TOKEN_TRACE("Already in the cache");
			continue;
		}

		p = ( p == NULL ) ? malloc( sizeof(pgpTokenKeyInfoPriv) ) :
			realloc( p, (n+1)*sizeof(pgpTokenKeyInfoPriv) );

		if( !sGetAllKeyInfo( tptr, &pub, (pgpTokenKeyInfoPriv*)p+n,  
								  &data_buf, &data_buf_size ) )
			break;

		/* No PGP DATA for this keyid */
		((pgpTokenKeyInfoPriv*)p+n)->pgpData = FALSE;

		n++;
	}

	free( handles );
	free( data_buf );

	if( i != howmany )  {	/* We had some errors */
		free( p );
		return NULL;
	}

	PGP_TOKEN_TRACE1("%d browser-generated objects found", n-p_n);

	*n_out = n-p_n;
	return (PGPTokenKeyInfo*)p;
}

static void sTrim( PGPByte *s, int size )  {
    PGPByte *p;
    
    if( *s == '\0' )
        return;

    for( p = s + size-1; p >= s; p-- )  {
        if( *p == ' ' )
            *p = '\0';
        else
            break;
    }

    return;
}

static PGPBoolean
sP11TokenInfo( const PGPUInt32 slot, PGPTokenInfo *info )
{
	CK_RV rv;
	CK_TOKEN_INFO tinfo;

#if PGP_DEBUG
    CK_SLOT_INFO sinfo;

    memset( &sinfo, 0, sizeof(sinfo) );
    rv = F->C_GetSlotInfo(slot, &sinfo);
    if( rv != CKR_OK  ) {
        pgpDebugMsg( "C_GetSlotInfo failed" );
        return FALSE;
    }

    if( !(sinfo.flags & CKF_TOKEN_PRESENT) )   {
        PGP_TOKEN_TRACE( "sP11TokenInfo: token is not present" );
        return FALSE;
    }
#endif

    memset( &tinfo, 0, sizeof(tinfo) );
    rv = F->C_GetTokenInfo(slot, &tinfo);
    if( rv != CKR_OK ) {
        pgpDebugMsg( "C_GetTokenInfo failed" );
        return FALSE;
    }

    info->size = sizeof( *info );

    sTrim( tinfo.manufacturerID, sizeof(tinfo.manufacturerID) );
    sTrim( tinfo.model, sizeof(tinfo.model) );
    sTrim( tinfo.serialNumber, sizeof(tinfo.serialNumber) );

    strncpy( info->manufacturerID, tinfo.manufacturerID, sizeof(info->manufacturerID)-1 );
    info->manufacturerID[sizeof(info->manufacturerID)-1] = '\0';
    strncpy( info->model, tinfo.model, sizeof(info->model)-1 );
    info->model[sizeof(info->model)-1] = '\0';
    strncpy( info->serialNumber, tinfo.serialNumber, sizeof(info->serialNumber)-1 );
    info->serialNumber[sizeof(info->serialNumber)-1] = '\0';

    info->minPinLen = tinfo.ulMinPinLen;
    info->maxPinLen = tinfo.ulMaxPinLen;

    return TRUE;
}

/* Get the range for the keysize, supported by the RSA key generation 
   mechanism
 */
PGPBoolean static 
sGetRsaKeySizeRange(PGPUInt32 slot, PGPTokenInfo *info)
{
	CK_RV rv;
	int nMechanisms=0;
	int nMechanisms2;
	CK_MECHANISM_TYPE *pMechList = NULL;
	int i;

	info->bRsa = FALSE;
	info->minRsaKeySize = info->maxRsaKeySize = 0;

	rv = F->C_GetMechanismList(slot, (CK_MECHANISM_TYPE_PTR)NULL, &nMechanisms);
	if( nMechanisms == 0 )
		return FALSE;

	pMechList = calloc( sizeof(CK_MECHANISM_TYPE), nMechanisms );
	if( pMechList == NULL )
		return FALSE;

	rv = F->C_GetMechanismList(slot, pMechList, &nMechanisms2);
	if( rv != CKR_OK )  {
		free( pMechList );
		return FALSE;
	}

	pgpAssert(nMechanisms == nMechanisms2);


	for( i=0; i<nMechanisms; i++ )  {
		/* Get only RSA for now */
		if( pMechList[i] == CKM_RSA_PKCS_KEY_PAIR_GEN )  {
			break;
		}
	}
	free( pMechList );

	if( i == nMechanisms )
		return TRUE; /* No key generation. Card can still be used in read-only mode */

	/* Card supports CKM_RSA_PKCS_KEY_PAIR_GEN. Get deatils. */
	{
		CK_MECHANISM_INFO minfo;

		(void)memset(&minfo, 0, sizeof(CK_MECHANISM_INFO));
		rv = F->C_GetMechanismInfo(slot, CKM_RSA_PKCS_KEY_PAIR_GEN, &minfo);
		if( rv != CKR_OK ) {
			pgpDebugMsg( "C_GetMechanismInfo failed for supported mechanism" );
			return FALSE;
		}

		/* Double check */
		if( !( minfo.flags & CKF_GENERATE_KEY_PAIR ) )  {
			pgpDebugMsg( "Inconsistency: mechanism expected to "
				"allow the key generation, but appropriate flag is not set" );
			return FALSE;
		}

		info->bRsa = TRUE;
		info->minRsaKeySize = minfo.ulMinKeySize;
		info->maxRsaKeySize = minfo.ulMaxKeySize;
	}

	return TRUE;
}

static CK_RV sP11OpenSession( PGPToken *tptr ) {
	CK_RV rv;

	if (tptr->have_slot)
		rv = F->C_OpenSession(tptr->slot, CKF_RW_SESSION | CKF_SERIAL_SESSION,
			                  0/*tptr*/, NULL/*sP11Callback*/, &(tptr->session));
	else return CKR_SLOT_ID_INVALID;

	if (rv != CKR_OK)	{
		tptr->session = 0;
		tptr->have_session = FALSE;
		PGP_TOKEN_TRACE( "C_OpenSession failed" );

		return rv;
	}

	tptr->have_session = TRUE;
	return CKR_OK;
}

/* Copies KEYIDs from the cache to the buffer to be freed by the 
   caller. 
   Implemenation of this finction can be made dummy, if copying 
   not needed later. 
 */
static PGPTokenKeyInfo *sDupKeyIDs( const PGPTokenKeyInfo *in, int n )  
{
	pgpTokenKeyInfoPriv *out;

	pgpAssert( n && in );
	out = calloc( sizeof( pgpTokenKeyInfoPriv ), n );

	if( out )
		memcpy( out, in, n * sizeof( pgpTokenKeyInfoPriv ) );

	return (PGPTokenKeyInfo*)out;
}

/* Prepare key infos in the ascending order of the first 
   sizeof(unsigned) bytes of keyid */
static void sPrepareForCache( PGPToken *tptr, 
	pgpTokenKeyInfoPriv *ki, int n )
{
	pgpTokenKeyInfoPriv last;
	pgpTokenKeyInfoPriv *p_last;
	unsigned last_value;

	unsigned curr;

	int changed;

	int i,j;

	if( n == 0 || ki == NULL )  {
		PGP_TOKEN_TRACE("sPrepareForCache: nothing to prepare");
		return;
	}

	tptr->cachedTimeStamp = -1;

	/* Sort key infos */
	for( i=0; i<n; i++ )  {
		p_last = ki+i;
		last_value = *(unsigned*)p_last->keyid;
		changed = FALSE;

		for( j=i+1; j<n; j++ )  {
			curr = *(unsigned*)(ki+j)->keyid;
			
			if( curr < last_value )  {
				last_value = curr;
				p_last = ki+j;
				changed = TRUE;
			}
		}

		/* swap */
		if( changed ) {
			memcpy( &last, p_last, sizeof(last) );

			memmove( p_last/*dst*/, ki+i/*src*/, sizeof(*p_last) );
			memcpy( ki+i/*dst*/, &last, sizeof(last) );

			i--;
		}
	}

	pgpAssert( sCacheIsSorted(ki, n) );
}

static void sInvalidateCache( PGPToken *tptr )  {
	free( tptr->privKeyIDs );
	free( tptr->pubKeyIDs );
	tptr->privKeyIDs = tptr->pubKeyIDs = NULL;
	tptr->nPrivKeyIDs = tptr->nPubKeyIDs = 0;
	tptr->cachedTimeStamp = 0;
	PGP_TOKEN_TRACE("Cache invalidated");
}

#if 0
/* PKCS11 callback -- for debug purposes */
static CK_RV CK_ENTRY sP11Callback(
  CK_SESSION_HANDLE hSession,     /* the session's handle */
  CK_NOTIFICATION   event,
  CK_VOID_PTR       pApplication  /* same as passed to C_OpenSession. */ )
{
	PGPToken *tptr = (PGPToken *)pApplication;

	(void)hSession;
	(void)event;

#if PGP_DEBUG
	switch( event )  {
	case CKN_SURRENDER:
		PGP_TOKEN_TRACE("CKN_SURRENDER");
		break;
	case CKN_COMPLETE:
		PGP_TOKEN_TRACE("CKN_COMPLETE");
		break;
	case CKN_DEVICE_REMOVED:
		PGP_TOKEN_TRACE("CKN_DEVICE_REMOVED");
		break;
	default:
		PGP_TOKEN_TRACE("?");
		pgpAssert(0);
		break;
	}
#endif

	return CKR_OK;
}
#endif


/* Find all private keys on the card by reading CKO_PUBLIC_KEY objects. 
   This will work for PGP-generated and browser-generated cards.

   Caller must free returned array n_out PGPTokenKeyInfos.

   There are more or equal "private keys" then "public"
*/
static PGPTokenKeyInfo* 
sGetPrivKeyIDs(PGPToken *tptr, PGPSize *n_out, PGPBoolean is_dup )
{
	int howmany=0;
	CK_OBJECT_HANDLE *handles=NULL;

	const CK_OBJECT_CLASS key = CKO_PUBLIC_KEY;	
		/* We use public key, because it doesn't require PIN */
	const CK_KEY_TYPE keyType = CKK_RSA;
	CK_ATTRIBUTE templRsa[] =
	{
		{ CKA_CLASS,	(CK_OBJECT_CLASS *)&key,	sizeof(key) },
		{ CKA_KEY_TYPE, (CK_KEY_TYPE *)&keyType ,	sizeof(keyType ) },
	};

	void *buffer=NULL;
	int buffer_size=0;

	pgpTokenKeyInfoPriv *p_out_temp=NULL;
	int p_out_temp_size=0;

	int i;

	*n_out = 0;

	if (!tptr->have_session)
	{
		pgpDebugMsg( "sGetPrivKeyIDs called with no smartcard session" );
		return NULL;
	}

	sCheckCache( tptr );

	if( tptr->privKeyIDs )  {
		PGPTokenKeyInfo *out = 
			is_dup ? sDupKeyIDs( tptr->privKeyIDs, 
			tptr->nPrivKeyIDs ) : tptr->privKeyIDs;
		pgpAssert( sCacheIsSorted((pgpTokenKeyInfoPriv*)tptr->privKeyIDs, tptr->nPrivKeyIDs) );

		if( out )  
			*n_out = tptr->nPrivKeyIDs;
		return out;
	}

	/* IDs were not cached. Read data from the card. */
	handles = sFindP11Objs( tptr, templRsa, sizeof(templRsa), &howmany, NULL, 0 );

	if( handles == NULL )
		return NULL;

	for( i=0; i<howmany; i++ )  {
		pgpTokenKeyInfoPriv input;

		memset( &input, 0, sizeof(input) );
		input.handle = handles[i];

		p_out_temp = (p_out_temp==NULL) ? 
			(pgpTokenKeyInfoPriv*)malloc( sizeof(pgpTokenKeyInfoPriv) ) :
			(pgpTokenKeyInfoPriv*)realloc( p_out_temp, 
							(p_out_temp_size+1)*sizeof(pgpTokenKeyInfoPriv) );

		if( p_out_temp == NULL )
			break;

		if( !sGetAllKeyInfo( 
				tptr, &input, p_out_temp+p_out_temp_size, 
				&buffer, &buffer_size ) 
				)  
		{
			pgpAssert(0);
			break;
		}

		p_out_temp_size ++;
	}

	free( handles );
	free( buffer );

	if( i != howmany ) {
		free( p_out_temp );
		return NULL;
	}

	/* Put private IDs into the cache */
	sPrepareForCache(tptr, p_out_temp, p_out_temp_size);

	tptr->nPrivKeyIDs = p_out_temp_size;
	tptr->privKeyIDs = (PGPTokenKeyInfo*)p_out_temp;
	{
		PGPTokenKeyInfo *out = is_dup ? 
			sDupKeyIDs( (PGPTokenKeyInfo*)p_out_temp, p_out_temp_size ) :
			(PGPTokenKeyInfo*)p_out_temp;
			
		if( out )  {
			*n_out = p_out_temp_size;
			PGP_TOKEN_TRACE1("sGetPrivKeyIDs: cache created for %d keys", p_out_temp_size );
		}
		return out;
	}
}

static PGPTokenKeyInfo* 
sGetPubKeyIDs(PGPToken *tptr, PGPSize *n, PGPBoolean is_dup)
{
	PGPTokenKeyInfo *out;
	PGPSize n_certs=0;

	if (!tptr->have_session)
	{
		pgpDebugMsg( "sGetPubKeyIDs called with no smartcard session" );
		return NULL;
	}

	pgpAssert( n );

	sCheckCache(tptr);

	if( tptr->pubKeyIDs )  {
		pgpAssert( tptr->nPubKeyIDs );

		out = is_dup ? 
			sDupKeyIDs( tptr->pubKeyIDs, tptr->nPubKeyIDs ) :
			tptr->pubKeyIDs;
		if( out )  {
			*n = tptr->nPubKeyIDs;
			pgpAssert( sCacheIsSorted((pgpTokenKeyInfoPriv*)tptr->pubKeyIDs, tptr->nPubKeyIDs) );
		}
		return out;
	}

	/* IDs were not cached. Read data from the card. */

	out = sGetPubKeyIDsFromPgpData( tptr, n );
	sPrepareForCache( tptr, (pgpTokenKeyInfoPriv*)out, *n );

	out = sGetPubKeyIDsFromCerts( tptr, out, *n, &n_certs );
	*n += n_certs;

	/* Put public IDs into cache */
	sPrepareForCache( tptr, (pgpTokenKeyInfoPriv*)out, *n );
	if( out )  {
		tptr->nPubKeyIDs = *n;
		tptr->pubKeyIDs = out;

		if( out == NULL )  
			*n = 0;
		else  {
			if( is_dup )  {
				if( (out = sDupKeyIDs( out, *n )) == NULL )
					*n = 0;
			}
			pgpAssert( sCacheIsSorted((pgpTokenKeyInfoPriv*)out, *n) );
			PGP_TOKEN_TRACE1("sGetPubKeyIDs: cache created for %d keys", *n);
		}
	}

	return out;
}


/* Puts key container for the key pair, 
   corresponding to keyID, or returns key container.

   uuid can be NULL, or not NULL. 
   if uuid is not NULL, 
		uuid on the input contains UUID of the containter that 
		the caller wants to create. 
   
		If valid containers exist for the keyID, they will 
		be destroyed. 
	
	uuidOut can be NULL or not NULL. 
	if uuidOut is not NULL, 
		on the output uuidOut contains UUID of existing 
		key container (possibly just created). 

   This is designed to import X509 certificates into Internet Explorer. 

   This implementation uses format, understandable by iKey.
   For other tokens we still use the same (straightforward) format to store
   key container information, otherwise not visible in the PKCS11
   object model. So, you should realize that technically, we create a key 
   container only for iKey. For other cards, we create something 
   like reflection of key container.

   This is the only CryptoAPI-related piece in this file. One day vendors 
   may provide other means for reading key container for existing keypair
   in the PKCS11 object model in the way, similiar to the iKey. 
*/
static PGPError 
sPutGetKeyContainer( PGPToken *tptr,
        const PGPByte *keyID, 
		PGPBoolean destroyExisting, 
        const PGPByte *uuid, PGPSize uuid_size, 
		PGPByte **uuidOut, PGPSize *uuidOutSize )
{
#ifdef PGP_WIN32
    /* Used for IExplorer only */
    PGPError ret = kPGPError_SmartCardError;
	CK_RV rv;
	CK_OBJECT_HANDLE *handles = NULL;
	CK_OBJECT_HANDLE handle;
	CK_BBOOL True = TRUE;
	CK_BBOOL False = FALSE;
	CK_OBJECT_CLASS data = CKO_DATA;
    PGPByte newID_padded[ID_PADDED_LEN] = { '\0' }; 
	PGPBoolean dontCreate = FALSE;
    unsigned howmany;
	CK_ATTRIBUTE srchTemplate[] =
	{
		{ CKA_CLASS, &data, sizeof(data) },
		{ CKA_TOKEN, &True, 1 },
        { CKA_PRIVATE, &False, 1 }, 
		{ CKA_APPLICATION, "v2.0", 4 },
        /* Search part ends here */

		{ CKA_LABEL, (void*)uuid, uuid_size },
		{ CKA_VALUE, newID_padded, sizeof(newID_padded) }
	};
    const srchTemplateSize = sizeof(srchTemplate)/sizeof(srchTemplate[0]);

	/* This is true for Windows: */
    pgpAssert( uuid_size==0 || uuid_size >= 0xE );   

    if( uuidOut != NULL )
        *uuidOut = NULL;
    if( uuidOutSize != NULL )
        *uuidOutSize = 0;

	PGP_TOKEN_TRACE2( "sPutGetKeyContainer(%x,%s)", *(unsigned*)keyID, uuid ? uuid : "NULL" );

	/* Do some consistency checks first... */
    if( ! sKeyIDToPubKeyInfo( tptr, keyID, NULL ) )  {
		PGP_TOKEN_TRACE( "sPutGetKeyContainer specifies unknown keyID. Exiting..." );
        return kPGPError_SmartCardError;    
            /* No keyID on this card -- will not create key container */
	}

	if (!(tptr->logged_in))
	{
		pgpDebugMsg( "sPutGetKeyContainer being called when not logged in" );
		return kPGPError_SmartCardKeyNotFound;
	}

    memcpy( newID_padded, keyID, KEYIDLEN );

    /* Search for all data objects. We do not put ID here, because
	   we want to support all containers: with KEYIDLEN, 
	   or ID_PADDED_LEN ID length. One vendor is sensitive 
	   for exact match for the attribute length specified in the 
	   template to the one actually on the token. 
	*/
    handles = sFindP11Objs( tptr, srchTemplate, 
		sizeof(srchTemplate)-2*sizeof(srchTemplate[0]), &howmany, NULL, 0 );

	if( handles != NULL )
	{
        unsigned i=0;
        PGPByte id[ID_PADDED_LEN] = { '\0' };  /* No more then ID_PADDED_LEN, OK to fail if more */
        PGPByte *curr_uuid = NULL;
		PGPSize curr_uuid_len = 0;
	    CK_ATTRIBUTE t[] =
	    {
		    { CKA_LABEL, NULL, 0 },	/* UUID */
		    { CKA_VALUE, NULL, 0 }  /* ID, bounded by ID_PADDED_LEN */
	    };

        /* Compare KeyID */
        for( i=0; i<howmany; i++ )  {
			t[0].pValue = NULL;
			t[1].pValue = NULL;

	        rv = F->C_GetAttributeValue(tptr->session, handles[i], t, 2);
	        if (rv != CKR_OK) {
		        pgpDebugMsg( "C_GetAttributeValue failed" );
                free(handles);
                return kPGPError_SmartCardError;
            }

            /* Set CKA_LABEL */
            pgpAssert( t[0].type == CKA_LABEL );
			if( curr_uuid_len < t[0].ulValueLen )  {
				curr_uuid_len = t[0].ulValueLen;
				curr_uuid = curr_uuid ? realloc( curr_uuid, curr_uuid_len ) :
					malloc( curr_uuid_len );

				if( curr_uuid == NULL )    {
					free(handles);
					return kPGPError_OutOfMemory;
				}
			}
            t[0].pValue = curr_uuid;
			*curr_uuid = '\0';	/* Make sure -- we reuse buffer */

            /* Set CKA_VALUE */
            pgpAssert( t[1].type == CKA_VALUE );
            if( t[1].ulValueLen > ID_PADDED_LEN )
                continue;
            t[1].pValue = id;   /* ID is stored in CKA_VALUE */

	        rv = F->C_GetAttributeValue(tptr->session, handles[i], t, 2);
	        if (rv != CKR_OK) {
		        pgpDebugMsg( "C_GetAttributeValue failed" );
		        continue;   /* Allowed fail */
            }

            if( memcmp( keyID, id, KEYIDLEN ) == 0 ) {  /* KeyID match */
				/* Here, we beleive we have a key container. Either return it, 
				   or destroy */
				if( destroyExisting )  {
					if(howmany == 1 && uuid && memcmp( curr_uuid, uuid, uuid_size )==0)  
					{
						/* We have exactly what needed. Do not delete/create */
						break;
					}
					PGP_TOKEN_TRACE1( "sPutGetKeyContainer: Destroy container %s", curr_uuid );
					rv = F->C_DestroyObject( tptr->session, handles[i] );
					if (rv != CKR_OK) 
						pgpDebugMsg( "C_DestroyObject failed, but we are continuing" );
					continue;	
					/* Go through all valid containers, 
					in case there is a corruption */
				}
				else
					ret = kPGPError_NoErr;

                break;
            }
        }
        free( handles );

        if( curr_uuid[0] && i<howmany && IsntPGPError(ret) )  {
			pgpAssert( !destroyExisting ); 
						/* Otherwise, we should already destroy all of them */

			if( uuidOut )
				*uuidOut = curr_uuid;	/* pass allocated buffer to the caller */
			if( uuidOutSize )
				*uuidOutSize = curr_uuid_len;

			PGP_TOKEN_TRACE1( "sPutGetKeyContainer: return container %s", curr_uuid );
            return kPGPError_NoErr;
        }

		free( curr_uuid );
	}

	if( uuid == NULL )	/* Nothing was found and we cannot create new cont. */
		return kPGPError_NoErr;

	/* Ready to create key container */

	rv = F->C_CreateObject(tptr->session, srchTemplate, srchTemplateSize, &handle);
	if (rv != CKR_OK)  {
        pgpDebugMsg( "C_CreateObject failed" );
        return kPGPError_SmartCardOutOfMemory;
	}
    else  {
        ret = kPGPError_NoErr;
        if( uuidOut )  {
            *uuidOut = malloc( uuid_size );
            if( *uuidOut == NULL )  
                return kPGPError_OutOfMemory;
            
            memcpy( *uuidOut, uuid, uuid_size );
			PGP_TOKEN_TRACE1( "sPutGetKeyContainer: return container %s", uuid );
        }
#if PGP_DEBUG
		else
			PGP_TOKEN_TRACE1( "sPutGetKeyContainer: container %s created", uuid );
#endif

        if( uuidOutSize )
            *uuidOutSize = uuid_size;
    }

    return ret;
#else	/* PGP_WIN32 */
    return kPGPError_NoErr;
#endif
}

static PGPError sWipe(PGPToken *tptr, PGPByte *id, int id_size)
{
	CK_RV rv;
	PGPBoolean failed = FALSE;
	CK_BBOOL True = TRUE;
	CK_ATTRIBUTE templ[] = 
	{
		{ CKA_TOKEN, &True, 1 }, 
		{ CKA_ID, id, id_size }		/* should be second */
	};
	int templSize = sizeof(templ);
	CK_OBJECT_HANDLE handle;
	CK_ULONG howmany;
	int max_objects=128;	/* Sanity check for the case when C_DestroyObject fails */

	PGP_TOKEN_TRACE2("sWipe( %x, %d )", id ? *(unsigned*)id : -1, id_size );

	if (!tptr->have_session || !tptr->logged_in)
	{
		pgpDebugMsg( "Wipe called without smartcard login" );
		return kPGPError_SmartCardError;
	}

	if( id == NULL )
		templSize -= sizeof(templ[0]);

	do {
		/* One vendor failed when we first find all objects 
		   and then delete them one by one. 
		   It will fail on the second object.
		   The same vendor required extra find in V4 key generation code. 
		*/
		sFindP11Objs( tptr, templ, templSize, &howmany, &handle, 1 );
		if( howmany == 1 )  {
			rv = F->C_DestroyObject(tptr->session, handle);
			if (rv != CKR_OK)  {
				pgpDebugMsg( "C_DestroyObject failed, wipe proceeding..." );
				failed = TRUE;
			}
		}
		if( failed )
			max_objects--;

	} while( howmany && max_objects );

	return failed ? kPGPError_SmartCardError : kPGPError_NoErr;
}

#if 0
static PGPUInt32 sSimpleHash32( void *p, int bytes )  {
	PGPUInt32 out = 0;

	bytes /= sizeof(PGPUInt32);

	while( bytes-- )
		out ^= *(((unsigned*)p)++);

	return out;
}
#endif

/*** EXPORTABLE FUNCTIONS ***/

/* Return nonzero iff we found a change in the token list 
*/

#define ID_FROM_SLOT_ID(tptr, i) ~(i)

PGPInt32 pgpAcquireAllTokens(void)
{
	int i;
	static PGPUInt32 before_ids[MAXTOKENS] = { 0 };
	
	int slots=0;
	CK_RV rv;
	PGPUInt32 ret=0;

	int n_ids_matched=0;
	PGPUInt32 new_id=0;
	PGPUInt32 new_slot=0;

	if (!enable_checking) return 0;

	/* PGP_TOKEN_TRACE("pgpAcquireAllTokens"); */

	if (is_CRYPTOKI_initialized == 0)
	{
		PGP_TOKEN_TRACE("Calling C_Initialize(NULL)");
		pgpAssert( F );	/* Possibly, pgpTokenInit was not called */
		rv = F->C_Initialize(NULL);
		if (rv != CKR_OK)
		{
			pgpDebugMsg( "C_Initialize failed" );
			return 0;
		}
		PGP_TOKEN_TRACE("C_Initialize(NULL) OK");
		is_CRYPTOKI_initialized = 1;
	}

	rv = F->C_GetSlotList(TRUE, NULL, &slots);
	if (rv != CKR_OK)	{
		pgpDebugMsg( "C_GetSlotList failed" );
		return 0;
	}

	/* Nothing is present now -- clean and return */
	if (slots == 0)	{
		if( s_how_many_tokens == 0 ) 
			return 0;

		PGP_TOKEN_TRACE("All tokens removed");
		for (i = 0 ; i < MAXTOKENS ; i++)
			if (token[i].have_session) PKCS11Destroy(token+i);

		memset( before_ids, 0, sizeof(before_ids) );

		s_how_many_tokens = 0;
		return -1;
	}

	/* One vendor was ignoring given size and corrupting small buffer. 
	   Make sure we allocate enough space for all slots. */

	/* Expand buffer */
	if( now_slots_allocated < slots*sizeof(PGPUInt32) )  {
		now_slots = realloc( now_slots, 
			now_slots_allocated=slots*sizeof(PGPUInt32) );
	}
	if( now_slots == NULL )  {
		pgpAssert(0);
		return 0;
	}

	rv = F->C_GetSlotList(TRUE, now_slots, &slots);
	if (rv != CKR_OK)	{
		pgpDebugMsg( "C_GetSlotList failed" );
		return 0;
	}

	/* Some tokens -- see if there are changes. Slot IDs are unique in PKCS11 */
	for( i=0; i<slots; i++ )
	{
		int j=0;
		const PGPUInt32 id = ID_FROM_SLOT_ID(token+i, now_slots[i]);

		/* Check for the case when token was moved to another slot */
		for( j=0; j<MAXTOKENS; j++ )  {
			PGPToken t;
			
			/* Reorder, if possible */
			if( id == before_ids[j] )  {
				if( n_ids_matched != j )  {
					before_ids[j] = before_ids[n_ids_matched];
					before_ids[n_ids_matched] = id;
					
					memcpy( &t, token+j, sizeof(t) );
					memcpy( token+j, token+n_ids_matched, sizeof(*token) );
					memcpy( token+n_ids_matched, &t, sizeof(t) );
				}
				
				if( (++n_ids_matched) >= MAXTOKENS)  {
					s_how_many_tokens = MAXTOKENS;
					/* PGP_TOKEN_TRACE1("No changes: recognized all %d tokens", MAXTOKENS ); */
					return FALSE;	/* No changes: recognized all tokens */
				}

				break;	/* will go to next slot */
			}
		}
		
		/* new token with hash is in slot now_slots[i] */
		if( j==MAXTOKENS )   {
			PGP_TOKEN_TRACE1("Token in slot %x is new", now_slots[i] );
			new_id = id;
			new_slot = now_slots[i];
		}
	}

	/* Handle removed tokens (those which we didn't match with existing tokens) */
	for( i=n_ids_matched; i<MAXTOKENS; i++ )  {
		if (token[i].have_session) {
			PGP_TOKEN_TRACE2("One token removed (%d/%d)", 
				i-n_ids_matched, MAXTOKENS-n_ids_matched);

			PKCS11Destroy(token+i);
			ret = 1;
		}
		before_ids[i] = 0;
	}

	/* We have one new token to add. We add only one at a time.
	   n_ids_matched tokens already present */
	if( new_id && (i = n_ids_matched) < MAXTOKENS )  {
		PGP_TOKEN_TRACE("New token found");
		
		/* Add new token to the end of the list */
		PKCS11Init(token+i, new_slot, TRUE, i);
		
		before_ids[i] = new_id;
		ret = 1;

		s_how_many_tokens = n_ids_matched+1;
	}
	else  {
		s_how_many_tokens = n_ids_matched;
	}

	return ret;
}

void pgpReleaseAllTokens(void)
{
	PGPUInt32 i;
	HINSTANCE hModule;
	CK_C_GetFunctionList pC_GetFunctionList;

	PGP_TOKEN_TRACE("pgpReleaseAllTokens");
	
	/* Test that library is still loaded */
	if( (hModule = GetModuleHandle(CryptoKiDll)) != NULL )
	{
		pC_GetFunctionList = (CK_C_GetFunctionList)GetProcAddress(hModule,
			"C_GetFunctionList");
		if( pC_GetFunctionList != NULL )
		{
			if( (*pC_GetFunctionList)(&F) == CKR_OK )
			{
				for (i = 0 ; i < MAXTOKENS ; i++)
					if (token[i].have_session) token[i].destroy(&token[i]);
				PGP_TOKEN_TRACE( "Calling C_Finalize(0)" );
				F->C_Finalize(0);
				F = NULL;
			}
		}
	}

	is_CRYPTOKI_initialized = 0;
	s_how_many_tokens = -1;		/* don't know */
	enable_checking = TRUE;

	free(now_slots);
	now_slots = NULL;

	memset( token, 0, sizeof(token) );

	FreeLibrary( hModule );	
}

PGPError pgpTokenInit(void)
{
	pgpAssert( sizeof(pgpTokenKeyInfoPriv) == sizeof(PGPTokenKeyInfo) );
	return PKCS11ConstructAllTokenFields();
}

PGPUInt32 pgpCountTokenObjects(void)
{
	return (s_how_many_tokens <= 0) ? 0 : s_how_many_tokens;
}

PGPToken *pgpGetNthTokenObject(PGPUInt32 x)
{
	if( (int)x >= s_how_many_tokens || (int)x < 0 )
	{
		pgpDebugMsg( "Asking for token which doesn't exist" );
		return NULL;
	}

	return &token[x];
}


PGPToken *pgpGetTokenObjectByPrivKeyID(const PGPByte *keyID)
{
	int i = 0, j = 0;
	PGPToken *tptr;

	for (i = 0 ; i < s_how_many_tokens ; i++)
	{
		tptr = pgpGetNthTokenObject(i);
		j = sKeyIDToPrivKeyInfo( tptr, keyID, NULL, TRUE );
		if (j) return tptr;
	}	
	return NULL;
}

/*
PGPToken *pgpGetTokenObjectByPubKeyID(const PGPByte *keyID)
{
	int i = 0, j = 0;
	PGPToken *tptr;

	pgpAssert(0);	

	for (i = 0 ; i < s_how_many_tokens ; i++)
	{
		tptr = pgpGetNthTokenObject(i);
		if (tptr == NULL) return NULL;
		j = sKeyIDToPubKeyInfo(tptr, keyID, NULL);
		if (j) return tptr;
	}	
	return NULL;
}
*/

/*** STATIC PKCS11 FUNCTIONS ***/

static void PKCS11ConstructTokenFields(PGPToken *tptr)
{
	tptr->init				= &PKCS11Init;
	tptr->destroy			= &PKCS11Destroy;
	tptr->decrypt			= &PKCS11Decrypt;
	tptr->sign				= &PKCS11Sign;
	tptr->login				= &PKCS11Login;
	tptr->logout			= &PKCS11Logout;
	tptr->getPrivKeyIDs		= &PKCS11GetPrivKeyIDs;
	tptr->getPubKeyIDs		= &PKCS11GetPubKeyIDs;
	tptr->getPublicData		= &PKCS11GetPublicData;
    tptr->getX509Certs      = &PKCS11GetCerts;
    tptr->getX509Cert       = &PKCS11GetCert;
	tptr->putPublicData		= &PKCS11PutPublicData;
	tptr->putPrivate		= &PKCS11PutPrivate;
	tptr->del				= &PKCS11Delete;
	tptr->keygen			= &PKCS11Keygen;
	tptr->editKeyList		= &PKCS11EditKeyList;
	tptr->setPIN			= &PKCS11SetPIN;
	tptr->wipe				= &PKCS11Wipe;
    tptr->getInfo           = &PKCS11GetInfo;
    tptr->importX509        = &PKCS11ImportX509;
    tptr->putKeyContainer =   &PKCS11PutKeyContainer;
	tptr->getKeyContainer =   &PKCS11GetKeyContainer;
}

static PGPError PKCS11ConstructAllTokenFields(void)
{
	PGPUInt32 i = 0;
	CK_RV rv;
	HINSTANCE hModule;
	CK_C_GetFunctionList pC_GetFunctionList;
	s_how_many_tokens = -1;	/* don't know */

	PGP_TOKEN_TRACE("PKCS11ConstructAllTokenFields");

	hModule = LoadLibrary(CryptoKiDll);
	if (hModule == NULL)
	{
		pgpDebugMsg( "Target smartcard DLL not loaded" );
        enable_checking = FALSE;
		return kPGPError_SmartCardError;
	}

	pC_GetFunctionList = (CK_C_GetFunctionList)GetProcAddress(hModule,
		"C_GetFunctionList");

	if (pC_GetFunctionList == NULL)
	{
		pgpDebugMsg( "C_GetFunctionList failed" );
		FreeLibrary(hModule);
		return kPGPError_SmartCardError;
	}

	rv = (*pC_GetFunctionList)(&F);

	for (i = 0 ; i < MAXTOKENS ; i++)
		PKCS11ConstructTokenFields(token+i);
	return kPGPError_NoErr;
}


static PGPError PKCS11Init(PGPToken *tptr, PGPUInt32 whichslot,
						   PGPBoolean haveslot, PGPInt32 index)
{
	tptr->logged_in = FALSE;
	tptr->slot = whichslot;
	tptr->have_slot = haveslot;
	tptr->tokenNum1 = index + 1;

	if( sP11OpenSession( tptr ) != CKR_OK )  {
		pgpDebugMsg( "sP11OpenSession failed" );
		return kPGPError_SmartCardError;
	}

	return kPGPError_NoErr;
}

static PGPError PKCS11Destroy(PGPToken *tptr)
{
	CK_RV rv;

	PGP_TOKEN_TRACE("PKCS11Destroy");

	if (tptr->logged_in)
	{
		rv = F->C_Logout(tptr->session);
		if (rv != CKR_OK)
		{
/*			pgpDebugMsg( "C_Logout failed" );  */
		}
	}

	if (tptr->have_session)
	{
		rv = F->C_CloseAllSessions(tptr->slot);
		if (rv != CKR_OK)
		{
/*			pgpDebugMsg( "C_CloseAllSessions failed" );  */
		}
	}

	free( tptr->privKeyIDs );
	free( tptr->pubKeyIDs );

	memset( tptr, 0, sizeof(*tptr) );
	PKCS11ConstructTokenFields( tptr );

	return kPGPError_NoErr;
}

static PGPError PKCS11Logout(PGPToken *tptr)
{
	CK_RV rv;

	PGP_TOKEN_TRACE("PKCS11Logout");

	if (!tptr->have_slot) return kPGPError_NoErr;
	if (!tptr->have_session) return kPGPError_NoErr;
	if (!tptr->logged_in) return kPGPError_NoErr;
	
	rv = F->C_Logout(tptr->session);
	if (rv != CKR_OK)
	{
		pgpDebugMsg( "C_Logout failed" );
	}

	tptr->logged_in = 0;

	return kPGPError_NoErr;
}

static PGPError PKCS11Login(PGPToken *tptr, char const *input, PGPSize length)
{
	CK_RV rv;

	PGP_TOKEN_TRACE("PKCS11Login");

    if (tptr->logged_in) {
        return kPGPError_NoErr;
    }

	rv = F->C_Login(tptr->session, CKU_USER, (void*)input, (CK_ULONG)length);
	if (rv == CKR_PIN_INCORRECT || rv == CKR_PIN_INVALID ||
		rv == CKR_PIN_LEN_RANGE )  
	{
		PGP_TOKEN_TRACE("Login failed: bad passphrase");
		return kPGPError_BadPassphrase;
	}

	if (rv != CKR_OK)
	{
		pgpDebugMsg( "C_Login failed" );
		return kPGPError_SmartCardError;
	}

	tptr->logged_in = 1;
	return kPGPError_NoErr;
}

static PGPError PKCS11Sign(PGPToken *tptr, CK_BYTE *keyID, CK_BYTE *in,
					  PGPSize isize, CK_BYTE *out, PGPSize osize)
{
	CK_RV rv;
	CK_OBJECT_CLASS kt1Class = CKO_PRIVATE_KEY;
	CK_KEY_TYPE kt1KeyType = CKK_RSA;
	CK_BYTE kt1ID[KEYIDLEN];
	pgpTokenKeyInfoPriv ki; 
	CK_MECHANISM mechanism = { CKM_RSA_PKCS, NULL_PTR, 0 };
	CK_BYTE *bigbuf;
	PGPUInt32 signatureSize;

	PGP_TOKEN_TRACE("PKCS11Sign");

	/* Buffer maintenance. */
	memcpy(kt1ID, keyID, KEYIDLEN);

	if (!(tptr->logged_in))
	{
		pgpDebugMsg( "C_Sign being called when not logged in" );
		return kPGPError_SmartCardKeyNotFound;
	}

	if ( !sKeyIDToPrivKeyInfo( tptr, kt1ID, &ki, TRUE ))  {
		return kPGPError_SmartCardKeyNotFound;
	}

	if( !(bigbuf = calloc( osize, 1 )) )	/* zeroes */
		return kPGPError_OutOfMemory;

	rv = F->C_SignInit(tptr->session, &mechanism, ki.handle);
	if (rv != CKR_OK)
	{
		pgpDebugMsg( "C_SignInit failed" );
		free(bigbuf);
		return kPGPError_SmartCardError;
	}

	signatureSize = osize;
	rv = F->C_Sign(tptr->session, in, isize, bigbuf, &signatureSize);
	if (rv != CKR_OK)
	{
		pgpDebugMsg( "C_Sign failed" );
		free(bigbuf);
		return kPGPError_SmartCardError;
	}

	if (signatureSize > osize) return kPGPError_SmartCardError;
	memset(out, 0, osize);
	memcpy(out + osize - signatureSize, bigbuf, signatureSize);

	free(bigbuf);

	return (PGPError)signatureSize;
}

static PGPError PKCS11Decrypt(PGPToken *tptr, CK_BYTE *keyID, CK_BYTE *in,
						 PGPSize isize, CK_BYTE *out, PGPSize osize)
{
	CK_RV rv;
	CK_OBJECT_CLASS kt1Class = CKO_PRIVATE_KEY;
	CK_KEY_TYPE kt1KeyType = CKK_RSA;
	CK_BYTE kt1ID[KEYIDLEN];
	pgpTokenKeyInfoPriv ki; 
	CK_MECHANISM mechanism = { CKM_RSA_PKCS, NULL_PTR, 0 };
	CK_BYTE *bigbuf;
	PGPUInt32 decryptSize;

	PGP_TOKEN_TRACE("PKCS11Decrypt");

	/* Buffer maintenance. */
	memcpy(kt1ID, keyID, KEYIDLEN);

	if (!(tptr->logged_in))
	{
		pgpDebugMsg( "C_Decrypt being called when not logged in" );
		return kPGPError_SmartCardKeyNotFound;
	}

	if( !sKeyIDToPrivKeyInfo( tptr, kt1ID, &ki, TRUE ) )
		return kPGPError_SmartCardKeyNotFound;

	if( !(bigbuf = calloc( osize, 1 )) )	/* zeroes */
		return kPGPError_OutOfMemory;

	rv = F->C_DecryptInit(tptr->session, &mechanism, ki.handle);
	if (rv != CKR_OK)
	{
		pgpDebugMsg( "C_DecryptInit failed" );
		free(bigbuf);
		return kPGPError_SmartCardError;
	}
	
	decryptSize = osize;
	rv = F->C_Decrypt(tptr->session, in, isize, bigbuf, &decryptSize);
	if (rv != CKR_OK)
	{
		pgpDebugMsg( "C_Decrypt failed" );
		free(bigbuf);
		return kPGPError_SmartCardError;
	}

	if (decryptSize > osize) return kPGPError_SmartCardError;

	memcpy(out, bigbuf, decryptSize);

	memset( bigbuf, 0xdd, osize );	/* clean sensitive data */
	free(bigbuf);

	return (PGPError)decryptSize;
}

static PGPTokenKeyInfo* 
PKCS11GetPrivKeyIDs(PGPToken *tptr, PGPSize *n)  {
	PGP_TOKEN_TRACE("PKCS11GetPrivKeyIDs");
	return sGetPrivKeyIDs( tptr, n, TRUE );
}

static PGPTokenKeyInfo* 
PKCS11GetPubKeyIDs(PGPToken *tptr, PGPSize *n)  {
	PGP_TOKEN_TRACE("PKCS11GetPubKeyIDs");
	return sGetPubKeyIDs(tptr, n, TRUE);
}


/*
	Get PGP Public Key Data, possibly with public key stub,
    which will allow to reconstruct public key packet. 

    di can be set to NULL, which means that returned value is ready 
    to be imported on the keyring (for now we always return di).
 */
static PGPByte* PKCS11GetPublicData(PGPToken *tptr, PGPByte *keyID, 
        PGPSize *size, pgpTokenDataInfo **di )
{
	PGPByte *value;
    PGPSize mapSize=0;
	pgpTokenKeyInfoPriv ki;


	PGP_TOKEN_TRACE("PKCS11GetPublicData");

    *di = NULL;
	
	if (!tptr->have_session)
	{
		pgpDebugMsg( "GetPublic called with no smartcard session" );
		return NULL;
	}

    /* Read PGP userIDs of the keyID, or X509 cert for other cards */
    if( !sKeyIDToPubKeyInfo(tptr, keyID, &ki) )  {
		PGP_TOKEN_TRACE1("No public data for %x", *(unsigned*)keyID);
        return NULL;
	}

    if( ! ki.pgpData )  {
		PGP_TOKEN_TRACE1("PKCS11GetPublicData: have NO PGP data for %x", *(unsigned*)keyID);
        return NULL; 
	}
	PGP_TOKEN_TRACE1("PKCS11GetPublicData: have PGP data for %x", *(unsigned*)keyID);

    /* Get content of PGP data */
    if( (value = sGetP11Value( tptr, ki.handle, &mapSize )) == NULL )  {
		pgpAssert("Could not read CKA_VALUE of the PGP data object");
        return NULL;
	}

    /* Get key stub information so the caller can reconstruct 
       the public key */
    {
        pgpTokenPubKeyStub *keyStub = (pgpTokenPubKeyStub *)value;
        
        /* Get CKO_PUBLIC_KEY values */
        PGPError err = sGetP11PubKey( tptr, keyID, keyStub, di );
        pgpAssert( IsntPGPError(err) );

        if( *di == NULL )  {
            /* For now we do not store modulus in the PGP DATA, so we will not be 
               able to reconstruct PGP public key packet. Later we might store 
               PGP public key as it is in the PGP DATA and delete CKO_PUBLIC_KEY. 
             */
            free(value);
            return NULL;
        }

        /* Get rid of key stub (later: make sure we had a key stub) */
        pgpAssert( mapSize > sizeof(pgpTokenPubKeyStub) );
        memmove( value, value + sizeof(pgpTokenPubKeyStub), mapSize-sizeof(pgpTokenPubKeyStub) );
        mapSize -= sizeof(pgpTokenPubKeyStub);
    }

	*size = mapSize;
	return value;
}

/* Get all X509 certificates for the given keyID. 

   Intended for both PGP and browser-generated tokens.
   (See sGetPubKeyIDsFromCerts)
*/
static PGPTokenCertInfo * 
PKCS11GetCerts(PGPToken *tptr, 
                   PGPTokenKeyInfo *keyInfo, PGPInt32 *n_out)
{
	CK_RV rv;

    CK_ATTRIBUTE idTemplate[] = {
		{ CKA_ID, NULL, 0 }
	};

    PGPByte *pID = NULL;
    PGPByte id_alloced_size = 0;

    /* CKA_ID of public key is conveniently stored before. 
       All returned certs must have the same CKA_ID */
    const PGPByte *id = ((pgpTokenKeyInfoPriv*)keyInfo)->id;
    const unsigned id_size = ((pgpTokenKeyInfoPriv*)keyInfo)->id_size;

	CK_OBJECT_CLASS certClass = CKO_CERTIFICATE;
	CK_CERTIFICATE_TYPE certType = CKC_X_509;
	CK_ATTRIBUTE certTemplate[] =
	{
		{ CKA_CLASS, &certClass, sizeof(certClass) },
		{ CKA_CERTIFICATE_TYPE, &certType, sizeof(certType) }
	};

	CK_OBJECT_HANDLE *handles = NULL;
	int howmany;
	int i;

    pgpTokenCertInfoPriv *p = NULL;
    int n=0;

    PGP_TOKEN_TRACE("PKCS11GetCerts");

    pgpAssert( sizeof( pgpTokenCertInfoPriv ) == sizeof( PGPTokenCertInfo ) );
	pgpAssert( id_size > 0 );
	pgpAssert( *(unsigned*)keyInfo->keyid != 0 );

	/* Find all cert. objects */
	handles = sFindP11Objs( tptr, certTemplate, sizeof(certTemplate), &howmany, NULL, 0 );
	if( handles == NULL )
		return NULL;

    /* Filter cert objects */
	for( i=0; i<howmany; i++ )  {
      	
        pgpAssert( idTemplate[0].type == CKA_ID );
        idTemplate[0].pValue = NULL;
        idTemplate[0].ulValueLen = 0;

	    rv = F->C_GetAttributeValue(tptr->session, handles[i], idTemplate, 1);
	    if (rv != CKR_OK) {
		    pgpDebugMsg( "C_GetAttributeValue failed" );
		    return NULL;
	    }

        if( idTemplate[0].ulValueLen < id_size )
            continue;

        if( idTemplate[0].ulValueLen > id_alloced_size )  {
            pID = pID ? malloc( idTemplate[0].ulValueLen ) : 
                realloc( pID, id_alloced_size + idTemplate[0].ulValueLen );
	        if( pID == NULL )
		        return NULL;
        }

	    idTemplate[0].pValue = pID;

	    rv = F->C_GetAttributeValue(tptr->session, handles[i], idTemplate, 1);
	    if (rv != CKR_OK) {
		    free(pID);
		    pgpDebugMsg( "C_GetAttributeValue failed" );
		    return NULL;
	    }

        if( memcmp( pID, id, id_size )==0 )  {
    
            /* Cert found */
		    p = ( p == NULL ) ? malloc( sizeof(pgpTokenCertInfoPriv) ) :
			realloc( p, (n+1)*sizeof(pgpTokenCertInfoPriv) );

            /* Store the handle so the caller will return this to us later */
            p[n].handle = handles[i];

            /* Save keyID for consistency checks */
            memcpy( p[n].keyid, keyInfo->keyid, sizeof(p[n].keyid) );

            n++;
        }
	}

	free( handles );
	free( pID );

	PGP_TOKEN_TRACE1("%d X509 certificates found", n);

	if( i != howmany )  {
		free( p );
		return NULL;
	}

	*n_out = n;
	return (PGPTokenCertInfo*)p;
}

static PGPByte *	   
PKCS11GetCert(
    PGPToken *tptr, PGPTokenCertInfo *certInfo, PGPSize *size)
{
    pgpAssert( sizeof( pgpTokenCertInfoPriv ) == sizeof( PGPTokenCertInfo ) );
    PGP_TOKEN_TRACE("PKCS11GetCert");

    return sGetP11Value( tptr, ((pgpTokenCertInfoPriv*)certInfo)->handle, 
        size );
}

static PGPError 
PKCS11PutPrivate(PGPToken *tptr, PGPBoolean is_master, TCL_PRIVATE_KEY_DATA_LIST)
{
	CK_RV rv;
	CK_BBOOL True = TRUE;
	CK_BBOOL False = FALSE;
	CK_OBJECT_CLASS privkey = CKO_PRIVATE_KEY;
	CK_KEY_TYPE rsa = CKK_RSA;
	CK_OBJECT_HANDLE handle_priv, handle_pub;
    PGPByte ID_padded[ID_PADDED_LEN] = { '\0' }; 
	CK_ATTRIBUTE privTemplate[] =
	{
		{ CKA_CLASS, &privkey, sizeof(privkey) },
		{ CKA_KEY_TYPE, &rsa, sizeof(rsa) },
		{ CKA_MODULUS, modulus, modulusSize },
		/* End of search part */

		{ CKA_TOKEN, &True, 1 },
		{ CKA_PRIVATE, &True, 1 },
		{ CKA_LABEL, 
			is_master ? labelPriv : labelPrivSub, 
			(is_master ? sizeof(labelPriv) : sizeof(labelPrivSub)) -1 },
		{ CKA_ID, ID_padded, sizeof(ID_padded) },					
		{ CKA_DECRYPT, &True, 1 },
		{ CKA_SIGN, &True, 1 },
		{ CKA_SIGN_RECOVER, &True, 1 }, 
		{ CKA_PUBLIC_EXPONENT, pubExp, pubExpSize },
		{ CKA_PRIVATE_EXPONENT, privExp, privExpSize },
		{ CKA_PRIME_1, prime1, prime1Size },
		{ CKA_PRIME_2, prime2, prime2Size },
		{ CKA_EXPONENT_1, exp1, exp1Size },
		{ CKA_EXPONENT_2, exp2, exp2Size },
		{ CKA_COEFFICIENT, coeff, coeffSize },
		{ CKA_UNWRAP, &True, 1 },

        /* Optional part: if you change it, change priv_size_min as well */
		{ CKA_SENSITIVE, &True, 1 },  /* Most critical in optional list */
        { CKA_EXTRACTABLE, &False, 1 },
		{ CKA_DERIVE, &False, 1 } // ?
	};

	CK_OBJECT_CLASS pubkey = CKO_PUBLIC_KEY;
	CK_ATTRIBUTE pubTemplate[] =
	{
		{ CKA_CLASS, &pubkey, sizeof(pubkey) },
		{ CKA_KEY_TYPE, &rsa, sizeof(rsa) },
		{ CKA_MODULUS, modulus, modulusSize },
        { CKA_PUBLIC_EXPONENT, pubExp, pubExpSize },
		{ CKA_LABEL, 
			is_master ? labelPub : labelPubSub, 
			(is_master ? sizeof(labelPub) : sizeof(labelPubSub)) -1 },
		{ CKA_ID, ID_padded, sizeof(ID_padded) },
		{ CKA_TOKEN, &True, 1 },
		{ CKA_PRIVATE, &False, 1 },
		{ CKA_WRAP, &True, 1 },

        { CKA_ENCRYPT, &True, 1 },
		{ CKA_VERIFY, &True, 1 },

        /* Optional part: if you change it, change pub_size_min as well */
		{ CKA_DERIVE, &False, 1 }, 
		{ CKA_VERIFY_RECOVER, &True, 1 }
	};

	PGPSize priv_size = sizeof(privTemplate) / sizeof(CK_ATTRIBUTE);
    const PGPSize priv_size_min = priv_size - 3;
	const PGPSize privTemplateSearch_size = 3;
	PGPSize pub_size = sizeof(pubTemplate) / sizeof(CK_ATTRIBUTE);
    const PGPSize pub_size_min = pub_size - 2;

	PGP_TOKEN_TRACE("PKCS11PutPrivate");


	if (!tptr->have_session || !tptr->logged_in)
	{
		pgpDebugMsg( "PutPrivate called without smartcard login" );
		return kPGPError_SmartCardError;
	}

	/* See if the key already exists */
	{
		int howmany=0;
		CK_OBJECT_HANDLE handle;
		sFindP11Objs( tptr, 
			privTemplate, sizeof(privTemplate[0]) * privTemplateSearch_size,
			&howmany, &handle, 1 );

		if( howmany == 1 )  {
			PGP_TOKEN_TRACE1( "PutPrivate: private key %x already on the token", *(PGPUInt32*)ID );
			return kPGPError_SmartCardKeyExists;
		}
	}

    memcpy( ID_padded, ID, KEYIDLEN );

    do  {
	    rv = F->C_CreateObject(tptr->session, privTemplate, priv_size, &handle_priv);
    }  while( rv == CKR_ATTRIBUTE_TYPE_INVALID && priv_size-- > priv_size_min );
	if (rv != CKR_OK)
	{
		pgpDebugMsg( "C_CreateObject failed for private key" );
		return kPGPError_SmartCardOutOfMemory;
	}

    do  {
	    rv = F->C_CreateObject(tptr->session, pubTemplate, pub_size, &handle_pub);
    }  while( rv == CKR_ATTRIBUTE_TYPE_INVALID && pub_size-- > pub_size_min );
	if (rv != CKR_OK)
	{
		pgpDebugMsg( "C_CreateObject failed for public key" );
        F->C_DestroyObject( tptr->session, handle_priv );
		return kPGPError_SmartCardOutOfMemory;
	}

	sInvalidateCache(tptr);

	return kPGPError_NoErr;
}

/*
	Put PGP Public Key Data, prefexed by keyStub
    Later: we may want to let keyStub=NULL, if pubKeyBlock is ready to put
*/
static PGPError 
PKCS11PutPublicData(PGPToken *tptr, PGPByte *keyID,
				PGPByte *pubKeyBlock, PGPSize size, 
                pgpTokenDataInfo *di )
{
	PGPUInt32 tmplSize, howmany;
    pgpTokenPubKeyStub *keyStub = &(di->pubKeyStub);
	CK_RV rv;
	CK_OBJECT_CLASS data = CKO_DATA;
	CK_BBOOL True = TRUE;
	CK_OBJECT_HANDLE handle;
	pgpPubKeyDataLabel label;
	CK_ATTRIBUTE srchTemplate[] =
	{
		{ CKA_LABEL, label, 0 },
		{ CKA_CLASS, &data, sizeof(data) },
		{ CKA_TOKEN, &True, 1 },
		{ CKA_APPLICATION, (void*)application, sizeof(application) - 1 },
		{ CKA_VALUE, NULL /*pubKeyBlock+keyStub*/, size+sizeof(*keyStub) }
	};
	PGPByte *pData=NULL;
	PGPError ret = kPGPError_NoErr;

	PGP_TOKEN_TRACE("PKCS11PutPublicData");

    pgpAssert( keyStub ); /* For now it is mandatory */

	tmplSize = sizeof(srchTemplate) / sizeof(CK_ATTRIBUTE);

	if (!tptr->have_session || !tptr->logged_in)
	{
		pgpDebugMsg( "PutPublic called without smartcard login" );
		return kPGPError_SmartCardError;
	}

	pData = malloc( sizeof(*keyStub) + size );
	if( pData == NULL )
		return kPGPError_OutOfMemory;

    pgpAssert( di->exp_size==0 && di->mod_size==0 );

	pgpAssert( srchTemplate[tmplSize-1].type == CKA_VALUE );
	srchTemplate[tmplSize-1].pValue = pData;

	memcpy( pData, keyStub, sizeof(*keyStub) );
	memcpy( pData+sizeof(*keyStub), pubKeyBlock, size );

	pgpAssert( srchTemplate[0].type == CKA_LABEL );
	sKeyIDToPublicLabel(keyID, &srchTemplate[0].ulValueLen, label );

	if( sFindP11Objs( tptr, srchTemplate, 
		sizeof(srchTemplate)-sizeof(srchTemplate[0]), &howmany, &handle, 1 ) != NULL )
	{
		CK_ATTRIBUTE newdata[] = 
		{
			{ CKA_VALUE, pData, size+sizeof(*keyStub) }
		};

		pgpAssert( howmany == 1 );

		rv = F->C_SetAttributeValue(tptr->session, handle, newdata, 1);
		if (rv != CKR_OK)
		{
			pgpDebugMsg( "C_SetAttributeValue failed" );
			ret = ( rv == CKR_DEVICE_MEMORY ) ? 
				kPGPError_SmartCardOutOfMemory : kPGPError_SmartCardError;
			goto end;
		}
	}
	else
	{
		rv = F->C_CreateObject(tptr->session, srchTemplate, tmplSize, &handle);
		if (rv != CKR_OK)
		{
			pgpDebugMsg( "C_CreateObject failed" );
			ret = kPGPError_SmartCardOutOfMemory;
			goto end;
		}
		sInvalidateCache(tptr);
	}

end:
	free(pData);
	return ret;
}

static PGPError PKCS11Keygen(PGPToken *tptr, PGPByte *ID,
	PGPSize modulusSize, PGPBoolean genMaster, 
	PGPByte *modulus, PGPUInt32 *pubExp)
{
	CK_OBJECT_HANDLE hPub, hPriv;
	CK_RV rv;
	PGPError err = kPGPError_NoErr;
	CK_MECHANISM mechanism = { 
		CKM_RSA_PKCS_KEY_PAIR_GEN, NULL_PTR, 0 };
	const CK_ULONG modBits = (CK_ULONG) modulusSize;
	const CK_ULONG modBytes = modulusSize/8;
	CK_BYTE publicExponent[] = { 0x01, 0x00, 0x01 };
	CK_BBOOL True = TRUE;
	CK_BBOOL False = FALSE;
	CK_ATTRIBUTE pubKeyTemplate[] =
	{
		{ CKA_LABEL, 
			genMaster ? labelPub : labelPubSub, 
			(genMaster ? sizeof(labelPub) : sizeof(labelPubSub)) - 1 },
		{ CKA_ID, ID, KEYIDLEN },					
		{ CKA_TOKEN, &True, 1 },
		{ CKA_ENCRYPT, &True, 1 },
		{ CKA_VERIFY, &True, 1 },
		{ CKA_WRAP, &True, 1 },
		{ CKA_MODULUS_BITS, (void*)&modBits, sizeof(modBits) },
		{ CKA_PUBLIC_EXPONENT, publicExponent, sizeof(publicExponent) }
	};
	CK_ATTRIBUTE privKeyTemplate[] =
	{
		{ CKA_TOKEN, &True, 1 },
		{ CKA_PRIVATE, &True, 1 },
		{ CKA_LABEL, 
			genMaster ? labelPriv : labelPrivSub, 
			(genMaster ? sizeof(labelPriv) : sizeof(labelPrivSub)) -1 },
		{ CKA_ID, ID, KEYIDLEN },					
		{ CKA_SENSITIVE, &True, 1 },
		{ CKA_DERIVE, &False, 1 },
		{ CKA_DECRYPT, &True, 1 },
		{ CKA_SIGN, &True, 1 },
		{ CKA_SIGN_RECOVER, &True, 1 }, 
		{ CKA_UNWRAP, &True, 1 },

        /* Bellow are attributes, which are desirable, but
           we can live without them if the token complains. 
           If you add (or move from the top) here, decrease szMinPrv */
        { CKA_MODIFIABLE, &True, 1 }, 
		{ CKA_EXTRACTABLE, &False, 1 }
	};
	CK_ULONG szPub = sizeof(pubKeyTemplate) / sizeof(CK_ATTRIBUTE);
	CK_ULONG szPrv = sizeof(privKeyTemplate) / sizeof(CK_ATTRIBUTE);
    CK_ULONG szMinPrv = szPrv - 2;

	PGP_TOKEN_TRACE("PKCS11Keygen");

	if (!tptr->have_session || !tptr->logged_in)
	{
		pgpDebugMsg( "KeyGen called without smartcard login" );
		return kPGPError_SmartCardError;
	}

	*pubExp = 0x10001;
    
    do {
	    rv = F->C_GenerateKeyPair(tptr->session, &mechanism, pubKeyTemplate, szPub,
			privKeyTemplate, szPrv, &hPub, &hPriv);
    } while( rv == CKR_ATTRIBUTE_TYPE_INVALID && --szPrv >= szMinPrv );
	if (rv != CKR_OK)  {
		pgpDebugMsg( "C_GenerateKeyPair failed" );
        return ( rv == CKR_DEVICE_MEMORY ) ? 
			kPGPError_SmartCardOutOfMemory : kPGPError_SmartCardError;
	}

	/* Find just generated public key. 
   	   Some vendors return hPub and hPriv backwards; 
	   others don't allow to get attributes using the hPub 
	 */
	{
		CK_OBJECT_HANDLE handle;
		int howmany;
		if( sFindP11Objs( tptr, pubKeyTemplate, sizeof(pubKeyTemplate[0])*2, &howmany, 
			&handle, 1 )==NULL )
		{
			pgpDebugMsg("sFindP11Objs failed for the public key of succesefuly generated pair");
			return kPGPError_SmartCardError;
		}
		pgpAssert( howmany == 1 );
		hPub = handle;
	}

	{
		CK_ATTRIBUTE attrPub[] = 
		{
			{ CKA_MODULUS, modulus, modBytes }
		};
		CK_ULONG attrSz = sizeof(attrPub) / sizeof(CK_ATTRIBUTE);
		
		memset( modulus, 0, modBytes );
		rv = F->C_GetAttributeValue(tptr->session, hPub, attrPub, attrSz);
		if (rv != CKR_OK)
		{
			memset( modulus, 0, modBytes );
			pgpDebugMsg( "C_GetAttributeValue failed" );
			return kPGPError_SmartCardError;
		}
	}

	sInvalidateCache(tptr);

	return err;
}

static PGPError PKCS11EditKeyList(PGPToken *tptr, PGPByte *origID,
	PGPByte *newID)
{
	CK_RV rv;
	CK_OBJECT_HANDLE handle;
	CK_OBJECT_CLASS key;
	CK_OBJECT_CLASS keys[] = {CKO_PUBLIC_KEY, CKO_PRIVATE_KEY};
    PGPByte newID_padded[ID_PADDED_LEN] = { '\0' }; 
	CK_ATTRIBUTE dataTemplate[] =
	{
		{ CKA_VALUE, NULL, 0 }
	};
	CK_ATTRIBUTE keySrchTmpl[] =
	{
		{ CKA_CLASS, &key, sizeof(key) },
		{ CKA_ID, origID, KEYIDLEN }
	};
	CK_ATTRIBUTE keyChangeTmpl[] =
	{
		{ CKA_ID, newID_padded, sizeof(newID_padded) }
	};
	PGPUInt32 sizeData = sizeof(dataTemplate) / sizeof(CK_ATTRIBUTE);
	PGPUInt32 szKeySrch = sizeof(keySrchTmpl) / sizeof(CK_ATTRIBUTE);
	PGPUInt32 szKeyChng = sizeof(keyChangeTmpl) / sizeof(CK_ATTRIBUTE);
	PGPUInt32 howmany, i;

	PGP_TOKEN_TRACE("PKCS11EditKeyList");

	if (!tptr->have_session)
	{
		pgpDebugMsg( "EditKeyList called without smartcard session" );
		return kPGPError_SmartCardError;
	}

    memcpy( newID_padded, newID, KEYIDLEN );
    pgpAssert( sizeof(newID_padded) >= KEYIDLEN );
    pgpAssert( sizeof(newID_padded)==KEYIDLEN || 
               newID_padded[KEYIDLEN] == '\0' );

	/*
		Now that we've got the keyID set in the list, change the
		original key's ID to match that of the list for private 
		and public PKCS11 key.
	*/

	for( i=0; i<sizeof(keys)/sizeof(keys[0]); i++ )  {
		key = keys[i];

		if( sFindP11Objs(
			tptr, keySrchTmpl, sizeof(keySrchTmpl), &howmany, &handle, 1 ) == NULL ) 
		{
			pgpDebugMsg( "sFindP11Objs failed" );
			return kPGPError_SmartCardError;
		}

		pgpAssert( howmany == 1 );
		
		rv = F->C_SetAttributeValue(tptr->session, handle, keyChangeTmpl, szKeyChng);
		if (rv != CKR_OK)
		{
			pgpDebugMsg( "C_SetAttributeValue failed" );
			sInvalidateCache(tptr);
			/* Should not increase used size of the token */
			return kPGPError_SmartCardError;
		}
	}

	sInvalidateCache(tptr);
	
	return kPGPError_NoErr;
}

static PGPError PKCS11SetPIN(PGPToken *tptr, char *oldPIN, PGPSize oldPINsize,
						char *newPIN, PGPSize newPINsize)
{
	CK_RV rv;

	PGP_TOKEN_TRACE("PKCS11SetPIN");

	if (!tptr->have_session)
	{
		pgpDebugMsg( "SetPIN called without smartcard session" );
		return kPGPError_SmartCardKeyNotFound;
	}

	rv = F->C_SetPIN(tptr->session, (CK_CHAR *) oldPIN, oldPINsize,
			(CK_CHAR *) newPIN, newPINsize);

	if (rv == CKR_PIN_INCORRECT || rv == CKR_PIN_INVALID ||
		rv == CKR_PIN_LEN_RANGE )
		return kPGPError_BadPassphrase;

	if (rv != CKR_OK)
	{
		pgpDebugMsg( "C_SetPIN failed" );
		return kPGPError_SmartCardError;
	}

	return kPGPError_NoErr;
}

static PGPError PKCS11Wipe(PGPToken *tptr)
{
	PGPError ret = sWipe( tptr, NULL, 0 );

    sInvalidateCache(tptr);

	return ret;

}

static PGPError PKCS11GetInfo( PGPToken *tptr, PGPTokenInfo *tokenInfo ) 
{
	PGPSize n;
    PGPBoolean r = TRUE;

	PGP_TOKEN_TRACE("PKCS11GetInfo");

    /* So far there is only one version of this structure, therefore: */
    pgpAssert( tptr->info.size == 0 /* Was error acquiring info */ ||
               tptr->info.size/*client size*/ == sizeof(*tokenInfo) );

	if( ! tptr->info.size )  {
		PGPBoolean r2;

		r = sP11TokenInfo( tptr->slot, &tptr->info );
#if PGP_DEBUG
		if( r )
			PGP_TOKEN_TRACE3( "PKCS11GetInfo: mid='%s', model='%s', SN='%s'", 
            tptr->info.manufacturerID, tptr->info.model, 
            tptr->info.serialNumber );
		else
			pgpDebugMsg( "sP11TokenInfo faled" );
#endif
		
		r2 = sGetRsaKeySizeRange( tptr->slot, &tptr->info );
#if PGP_DEBUG
		if( r2 )
			PGP_TOKEN_TRACE2( "PKCS11GetInfo: %d <= RSA_size <= %d", 
            tptr->info.minRsaKeySize, tptr->info.maxRsaKeySize );
		else
			pgpDebugMsg( "sGetRsaKeySizeRange faled" );
#endif

		r = (r & r2);
	}

    memcpy( tokenInfo, &tptr->info, sizeof(*tokenInfo) );

	/* Update dynamic fields */
	if( tptr->cachedTimeStamp == 0 )  {
		sGetPubKeyIDs( tptr, &n, FALSE );
		sGetPrivKeyIDs( tptr, &n, FALSE );
	}

	tokenInfo->numPubKeys = tptr->nPubKeyIDs;
	tokenInfo->numPrivKeys = tptr->nPrivKeyIDs;

	PGP_TOKEN_TRACE("PKCS11GetInfo returns");

    return r ? kPGPError_NoErr : kPGPError_SmartCardError;
}

/* Given values of all required fields for the PKCS11 cert object, 
   create CKO_CERTIFICATE object on the token */
static PGPError 
PKCS11ImportX509(PGPToken *tptr,        const PGPByte *keyID,
                 const PGPByte *userID,       PGPSize userID_len, 
                 const PGPByte *x509,         PGPSize x509_len, 
                 const PGPByte *subjDER,      PGPSize subjDER_len, 
                 const PGPByte *issuerDER,    PGPSize issuerDER_len, 
                 const PGPByte *sn,           PGPSize sn_len )
{
    PGPByte *label = NULL;
    unsigned label_size=0;
    PGPByte id_padded[ID_PADDED_LEN] = { '\0' };
	CK_BBOOL True = TRUE;
    CK_BBOOL False = FALSE;
	CK_OBJECT_CLASS certClass = CKO_CERTIFICATE;
	CK_CERTIFICATE_TYPE certType = CKC_X_509;

    CK_RV rv;
    CK_OBJECT_HANDLE handle;
    unsigned howmany=0;

    PGP_TOKEN_TRACE( "PKCS11ImportX509" );

    pgpAssert( keyID && ((PGPUInt32*)keyID)[1] != 0 );
    pgpAssert( x509 && x509_len );
    pgpAssert( subjDER && subjDER_len );
    pgpAssert( issuerDER && issuerDER_len );

    if( !tptr->logged_in )  {
        pgpDebugMsg( "PKCS11ImportX509: login failed" );
        return kPGPError_SmartCardKeyNotFound;
    }

    label = sKeyIDToP11CertLabel( keyID, userID, userID_len, &label_size );
    if( label == NULL )
        return kPGPError_OutOfMemory;

    memcpy( id_padded, keyID, KEYIDLEN );
    pgpAssert( KEYIDLEN <= sizeof(id_padded) );
    pgpAssert( KEYIDLEN == sizeof(id_padded) || 
               id_padded[KEYIDLEN] == '\0' );

    {
        CK_ATTRIBUTE templ[] =
        {
            { CKA_CLASS, &certClass, sizeof(certClass) },
            { CKA_ID, id_padded, sizeof(id_padded) },
            { CKA_LABEL, label, label_size },
            { CKA_ISSUER, (void*)issuerDER, issuerDER_len }, 
            { CKA_SERIAL_NUMBER, (void*)sn, sn_len }, 

            /* Search portion ends here (5 elements) */
            
            { CKA_CERTIFICATE_TYPE, &certType, sizeof(certType) },
            { CKA_SUBJECT, (void*)subjDER, subjDER_len }, 
            { CKA_VALUE, (void*)x509, x509_len },
            { CKA_TOKEN, &True, 1 },
            { CKA_PRIVATE, &False, 1 },

            /* Optional attributes are bellow. 
               If you add (move from the above) here, 
               decrement certMinAttrs. 
               Sort them from more important to less important */
            { CKA_MODIFIABLE, &True, 1 },
        };
        unsigned certAttrs = sizeof(templ)/sizeof(templ[0]);
        const unsigned certMinAttrs = certAttrs - 1;

        /* Search first for cert objects */
        sFindP11Objs( tptr, templ, 
            5 * sizeof(templ[0]), &howmany, &handle, 1 );
        if( howmany == 1 )  {
            PGP_TOKEN_TRACE("PKCS11ImportX509: cert already exist");
                /* I saw for some CA that it was unable to deferentiate 
                   between different certs for the same public key */
            free( label );
            return kPGPError_SmartCardX509Exists; /* Nothing to do */
        }

        /* Create certificate */
        do {
            rv = F->C_CreateObject(tptr->session, templ, certAttrs, &handle);
        } while( rv == CKR_ATTRIBUTE_TYPE_INVALID && --certAttrs >= certMinAttrs );
		if( rv == CKR_DEVICE_MEMORY )  {
			/* Certificate doesn't fit. Delete key conainer and try again. 
			   In general case information about key container will be lost. 
			 */
			PGP_TOKEN_TRACE1( "PKCS11ImportX509: cert doesn't fit:"\
				" delete all key containers for '%x'", *(PGPUInt32*)keyID );

			sPutGetKeyContainer( tptr, keyID, TRUE/*destroyExisting*/, NULL,0, NULL,NULL );
			rv = F->C_CreateObject(tptr->session, templ, certAttrs, &handle);
		}
	    if (rv != CKR_OK)  {
		    pgpDebugMsg( "C_CreateObject failed" );
            free( label );
            return ( rv == CKR_DEVICE_MEMORY ) ? 
				kPGPError_SmartCardOutOfMemory : kPGPError_SmartCardError;
	    }

        free( label );
    }

    /* Set/change CKA_SUBJECT for the private key (Netscape Requrement) 
       What if we have multiple certs for the same public key?
     */
    {
        CK_ATTRIBUTE keyChangeTmpl[] =
        {
            { CKA_SUBJECT, (void*)subjDER, subjDER_len }
        };
		pgpTokenKeyInfoPriv ki;

        /* Point handle to the corresponding private key */
        sKeyIDToPrivKeyInfo( tptr, keyID, &ki, TRUE /*private*/ );

		rv = F->C_SetAttributeValue(tptr->session, ki.handle, keyChangeTmpl, 1);
		if (rv != CKR_OK)  {
			pgpDebugMsg( "C_SetAttributeValue failed" );
            return ( rv == CKR_DEVICE_MEMORY ) ? 
				kPGPError_SmartCardOutOfMemory : kPGPError_SmartCardError;
		}
    }

    return kPGPError_NoErr;
}

static PGPError         
PKCS11PutKeyContainer( PGPToken *tptr,
                       const PGPByte *keyID, 
                       const PGPByte *uuid, PGPSize uuidSize )
{
	return sPutGetKeyContainer( tptr, keyID, TRUE/*destroyExisting*/, 
								uuid, uuidSize, NULL, NULL );
}

static PGPError         
PKCS11GetKeyContainer( PGPToken *tptr,
                       const PGPByte *keyID, 
                       PGPByte **uuidOut, PGPSize *uuidOutSize )
{
	return sPutGetKeyContainer( tptr, keyID, FALSE/*destroyExisting*/, 
								NULL, 0, uuidOut, uuidOutSize );
}

/* Delete related objects. 
   For V3 keys it means 
	delete public key, private key, data object [, cert, key container]. 
   For V4 --
    delete master pub key, priv key, data object, [cert, key container] 
    pub. subkey, priv. subkey
*/
static PGPError         
PKCS11Delete( PGPToken *tptr,
             const PGPByte *keyID, PGPBoolean is_last )
{
	PGPError ret;
	pgpTokenKeyInfoPriv ki;

	unsigned True = TRUE;
	pgpPubKeyDataLabel label;
	CK_ATTRIBUTE srchTemplate[] =
	{
		{ CKA_LABEL, label, 0 },
		{ CKA_TOKEN, &True, 1 },
		{ CKA_APPLICATION, (void*)application, sizeof(application) - 1 },
	};

	CK_OBJECT_HANDLE handle;
	int howmany=0;

	PGP_TOKEN_TRACE2("Delete: %x, %slast", *(unsigned*)keyID, is_last ? "" : "NOT ");

	if( !sKeyIDToPrivKeyInfo( tptr, keyID, &ki, FALSE ) )
		return FALSE;

	pgpAssert( ki.id && ki.id_size );

	ret = sWipe( tptr, ki.id, ki.id_size );

	/* Remove optional key container */
	sPutGetKeyContainer( tptr, ki.id, TRUE/*destroyExisting*/, NULL,0, NULL,NULL );

	/* Remove optional (for browser tokens) PGP data */
	sKeyIDToPublicLabel(keyID, &srchTemplate[0].ulValueLen, label );
	if( sFindP11Objs( tptr, srchTemplate, 
		sizeof(srchTemplate), &howmany, &handle, 1 ) )  
	{
	    CK_RV rv = F->C_DestroyObject(tptr->session, handle);
		if (rv != CKR_OK)  {
			pgpDebugMsg( "C_DestroyObject failed" );
			if( ret == kPGPError_NoErr )
				ret = kPGPError_SmartCardError;
		}
	}

#if PGP_DEBUG
	if( ret == kPGPError_NoErr )
		PGP_TOKEN_TRACE1("Delete: %x returns without error", *(unsigned*)keyID);
#endif

	if( is_last )
		sInvalidateCache(tptr);

	return ret;
}


PGPError pgpFreeMem( void *p )  {
    free(p);
    return kPGPError_NoErr;
}

PGPInt32 pgpGetTokenNum( PGPToken *tok )  {
	if( tok == NULL )
		return -1;

	PGP_TOKEN_TRACE1("pgpGetTokenNum: return token %d", tok->tokenNum1 );
	pgpAssert( tok->tokenNum1 > 0 );

	return tok->tokenNum1 - 1;
}

PGPError pgpSetP11DllName( const PGPByte *name )  
{
    PGP_TOKEN_TRACE1( "pgpSetP11DllName: setting PKCS11 DLL name as '%s'", name );
    if( !name || *name == '\0' )
        return kPGPError_BadParams;

    strncpy( CryptoKiDll, name, sizeof(CryptoKiDll)-1 );
    CryptoKiDll[sizeof(CryptoKiDll)-1] = '\0';

    enable_checking = TRUE;

    return kPGPError_NoErr;
}